<script type="text/javascript">
/**
 * フォーム送信時にデータを自動的に保存するための共通メソッド
 */
$(document).ready(function() {
    // すべてのフォームに対して送信イベントをバインド
    $('form').bind('submit', function(e) {
        // 対象フォームの情報
        const $form = $(this);
        const formId = $form.attr('id');
        const formName = $form.attr('name');
        
        // 現在のステップ名を取得（フォームIDからか、URLから）
        const stepName = formId || 'step' + window.location.pathname.split('step')[1];
        
        // フォーム要素の値とラベルを収集
        const formData = {};
        const formLabels = {};
        
        // input、select、textareaの値を収集
        $form.find('input, select, textarea').each(function() {
            const $input = $(this);
            const name = $input.attr('name');
            
            // nameがない、または送信ボタンはスキップ
            if (!name || $input.attr('type') === 'submit' || $input.attr('type') === 'button'|| $input.attr('type') === 'hidden') {
                return;
            }
            
            // input typeに応じた値の取得
            let value;
            if ($input.attr('type') === 'checkbox') {
                if ($input.attr('checked')) {
                    value = $input.next().text();
                } else {
                    return true; // チェックされていないラジオボタンはスキップ
                }
            } else if ($input.attr('type') === 'radio') {
                if ($input.attr('checked')) {
                    value = $input.next().text();
                } else {
                    return true; // チェックされていないラジオボタンはスキップ
                }
            } else if ($input.is('select')) {
                // select要素の場合、選択されたオプションのテキストを取得
                value = $input.find('option:selected').text();
            } else {
                value = $input.val();
            }
            if (value === undefined || value === null || value === '--') {
                return true;
            }
            
            formData[name] = ($input.attr('type') === 'checkbox' && formData[name]) ? (formData[name] + ';' + value) : value;
            
            // テーブル構造内の場合、対応するthを検索
            const $tr = $input.closest('tr');
            if ($tr.length) {
                const $th = $tr.find('th');
                if ($th.length) {
                    // ラベルテキストから※マークなどを削除してクリーンなテキストを取得
                    let labelText = $th.text().trim();
                    labelText = labelText.replace(/※|（.*?）|\(.*?\)/g, '').trim();
                    formLabels[name] = labelText;
                }
            } else {
                const $dd = $input.closest('dd');
                if ($dd.length) {
                    const $dt = $dd.prev('dt');
                    if ($dt.length) {
                        // dt要素からラベルを取得
                        let labelText = $dt.text().trim();
                        labelText = labelText.replace(/※|（.*?）|\(.*?\)/g, '').trim();
                        if (labelText.indexOf('\n') !== -1) {
                                if ($input.parent().is('li')) {
                                // dl要素内の場合、dt要素からラベルを取得
                                const $dt = $input.closest('ul').find('p');
                                if ($dt.length) {
                                    let labelText = $dt.text().trim();
                                    labelText = labelText.replace(/※|（.*?）|\(.*?\)/g, '').trim();
                                    formLabels[name] = labelText;
                                    return true; // ここでreturnすることで、ラベルの重複を防ぐ
                                }
                            }
                        }
                        formLabels[name] = labelText;
                    }
                } else {
                    if ($input.is('select')) {
                        const cts = $input.parent().contents();
                        if (cts.length > 0 && cts[0].nodeType === Node.TEXT_NODE) {
                            // 親要素のテキストノードからラベルを取得
                            let labelText = cts[0].textContent.trim();
                            labelText = labelText.replace(/※|（.*?）|\(.*?\)/g, '').trim();
                            formLabels[name] = labelText;
                            return true; // ここでreturnすることで、ラベルの重複を防ぐ
                        } 
                    }
                    const $label = $input.prevAll('p').length > 0 ? $input.prevAll('p') : $input.parent('p');
                    if ($label.length) {
                        // p要素からラベルを取得
                        let labelText = $label.text().trim();
                        labelText = labelText.replace(/※|（.*?）|\(.*?\)/g, '').trim();
                        formLabels[name] = labelText;
                    }
                }
            }
        });
        
        // 収集したデータをセッションストレージに保存
        const storageData = {
            values: formData,
            labels: formLabels,
            timestamp: new Date().toISOString()
        };
        
        // このステップのデータを保存
        sessionStorage.setItem(`${stepName}_data`, JSON.stringify(storageData));
        
        // すべてのステップのデータを更新
        let allStepsData = {};
        const savedAllData = sessionStorage.getItem('all_steps_data');
        if (savedAllData) {
            try {
                allStepsData = JSON.parse(savedAllData);
            } catch (e) {
                console.error('セッションデータの解析エラー:', e);
            }
        }
        
        // 現在のステップのデータを追加/更新
        allStepsData[stepName] = storageData;
        
        // 更新したデータを保存
        sessionStorage.setItem('all_steps_data', JSON.stringify(allStepsData));
        
        console.log(`${stepName}のデータを保存しました`, storageData);
        
        // フォーム送信を続行
        return true;
    });
});

/**
 * すべてのステップのフォームデータを取得する
 * @returns {Object} すべてのステップのデータ
 */
function getAllFormData() {
    const allData = sessionStorage.getItem('all_steps_data');
    return allData ? JSON.parse(allData) : {};
}

/**
 * 特定のステップのフォームデータを取得する
 * @param {string} stepName - ステップ名（例: 'step1'）
 * @returns {Object} 指定したステップのデータ
 */
function getStepFormData(stepName) {
    const stepData = sessionStorage.getItem(`${stepName}_data`);
    return stepData ? JSON.parse(stepData) : null;
}

/**
 * セッションストレージをクリア
 */
function clearFormSessionData() {
    // all_steps_dataのみを削除
    sessionStorage.removeItem('all_steps_data');
    
    // または、すべてのステップデータを削除する場合
    for (let i = 0; i < sessionStorage.length; i++) {
        const key = sessionStorage.key(i);
        if (key.includes('_data')) {
            sessionStorage.removeItem(key);
        }
    }
}

/**
 * 最終ステップでデータを整形して表示する
 * @param {string} containerId - 表示するコンテナのID
 */
function displayAllFormData(containerId) {
    const allData = getAllFormData();
    const $container = $('#' + containerId);

    const card_id = $('#card_number').val();
    
    if (!$container.length) {
        console.error(`ID ${containerId} の要素が見つかりません`);
        return;
    }
    
    const jsonObject = {data:[], card_id};
    // 各ステップのデータを整形
    Object.keys(allData).sort().forEach(stepName => {
        const stepData = allData[stepName];
        
        // 各フィールドの値とラベルを表示
        Object.keys(stepData.values).forEach(fieldName => {
            const value = stepData.values[fieldName];
            const label = stepData.labels[fieldName] || fieldName;

            
            // 値が有効な場合のみ表示（空でない、undefinedでない）
            if (value !== '' && value !== undefined && value !== null) {
                // jsonObject[label] = value;
                jsonObject.data.push({
                    label: label,
                    value: value
                });
            }
        });
    });
    $container.html(JSON.stringify(jsonObject));
}
</script>