<?php
/**
 * ファクトリー class
 *
 * @package Lambda/component
 * @author  rooth
 * @version 0.0.1
 *
 * PHP version 5
 *
 *<pre>
 *
 * usage:
 *  Factory::setBaseDir('/path/to/lib');
 *  $obj = Factory::load(str classPath [, str className])->getInstance([mixed arg1, arg2, ...]);
 *  $obj = Factory::load(str classPath [, str className])->getSingleton([mixed arg1, arg2, ...]);
 *  Factory::import(str classPath);
 *
 *   #クラスをロードするベースディレクトリを「/path/to/lib」にセット
 *   Factory::setBaseDir('/path/to/lib');
 *   // ※. このメソッドを呼ばない場合、デフォルトではこのクラスが置かれているディレクトリが設定されます。
 *
 *   # /path/to/lib/some_class.php をロードして、インスタンスを生成する
 *   $obj = Factory::load('some_class')->getInstance();
 *
 *   # /path/to/lib/hoge/some_class.php をロードして、シングルトンインスタンスを生成する
 *   $obj = Factory::load('hoge/some_class')->getSingleton();
 *
 *   # /path/to/lib/some_class.php をロードして、クラス名「className」のインスタンスを生成する
 *   $obj = Factory::load('some_class', 'className')->getInstance();
 *
 *   # /path/to/lib/hoge/some_class.php をロードして、コンストラクタ引数を渡してインスタンスを生成する
 *   $obj = Factory::load('some_class')->getInstance($arg1, $arg2, ...);
 *
 *   # /root/dir/some_class.php をロードして、インスタンスを生成する
 *   # ※ スラッシュから始まるパスを指定すると setBaseDir() で設定したディレクトリを無視する
 *   $obj = Factory::load('/root/dir/some_class')->getInstance();
 *
 *   # /root/dir/some_class.php をロードする ( = require)
 *   Factory::import('/root/dir/some_class');
 *
 *</pre>
 */

defined('DS') || define('DS', DIRECTORY_SEPARATOR);

class Factory
{
    protected $classname;
    protected static $singleton = array();
    protected static $base_dir;

    /**
     * ベースディレクトリをセットする
     *
     * @param  str  $dir ベースディレクトリ
     * @access public
     * @static
     */
    public static function setBaseDir($dir)
    {
        self::$base_dir = $dir;
    }

    /**
     * 1. クラスをロードする
     * 2. 自分自身をインスタンス化して返却する($load_only = false の時)
     * 
     * @param  str $path ロードするクラスパス
     *                   ・スラッシュで始まるパス以外は setBaseDir() で設定したディレクトリ配下からロードする
     *                   ・拡張子「.php」はあってもなくても良い
     *
     * @param  str  $classname クラスファイル名とクラス名が異なる場合に指定する
     * @param  bool $load_only true が指定された時、クラスファイルをloadして処理を終了する(self::importで使用)
     * @return obj 自分自身のインスタンス
     * @access public
     * @static
     */
    public static function load($path, $classname = '', $load_only = false)
    {
        if (self::$base_dir === null) self::setBaseDir(dirname(__FILE__));

        if (substr($path, -4) !== '.php') $path .= '.php';
        if (strpos($path, DS) !== 0)     $path  = rtrim(self::$base_dir, DS).DS.$path;

        require_once $path;

        // if $load_only === true then process exit.
        if ($load_only) return;

        if ($classname === '') {
            $arr = pathinfo($path);
            list($classname, ) = explode('.', $arr['basename']);
        }

        return new self($classname);
    }

    /**
     * クラスをロードする ( = require)
     *
     * @access public
     * @static
     */
    public static function import($path)
    {
        self::load($path, '', true);
    }

    /**
     * コンストラクタ
     * @access protected
     */
    protected function __construct($classname)
    {
        $this->classname = $classname;
    }

    /**
     * インスタンスを返却する
     *
     * @param  args コンストラクターに渡す引数(可変長)
     * @return obj
     * @access public
     */
    public function getInstance()
    {
        $args = func_get_args();

        if ( ! $args) return new $this->classname;

        $params = $this->mkParams($args);
        return eval("return new $this->classname($params);");
    }

    /**
     * シングルトンインスタンスを返却する
     *
     * @param  args コンストラクターに渡す引数(可変長)
     * @return obj
     * @access public
     */
    public function getSingleton()
    {
        $args = func_get_args();

        if ( ! isset(self::$singleton[$this->classname])) {
            $this->setSingleton($args);
        }
        return self::$singleton[$this->classname];

    }



    /**
     * @access protected
     */
    protected function setSingleton($args)
    {
        if ( ! $args) {
            self::$singleton[$this->classname] = new $this->classname;
        } else {
            $params = $this->mkParams($args);
            self::$singleton[$this->classname] = eval("return new $this->classname($params);");
        }
    }

    /**
     * @access protected
     */
    protected function mkParams(array $args)
    {
        $buff = array();
        for ($i = 0, $c = count($args); $i < $c; $i++) $buff[] = "\$args[$i]";
        return implode(',', $buff);
    }



} // --- End of Factory class


