<?php
/**
 * 家族歴 class
 *
 * @dependence DB/DAO_family_history.php
 */

class FamilyHistory
{
    protected $card_number;
    protected $now_datetime;
    protected $dao;

    /**
     * コンストラクタ
     *
     * @param string $card_number 診察券番号
     * @throw Exception
    */
    public function __construct($card_number)
    {
        $this->card_number  = $card_number;
        $this->now_datetime = date('Y-m-d H:i:s');
        $this->dao = Factory::load('DB/DAO_family_history')->getInstance();
    }

    /**
     * 家族歴の保存(追加、更新、削除)
     *
     * @param array $p HTTPポスト値の配列
     * @throw Exception
    */
    public function save(Array $p)
    {
        try {
            $this->upsert($p);
            $this->delete();
        } catch (Exception $e) {
            throw new Exception;
        }
    }

    /**
     * 家族歴のアップサート (レコードが存在すればupdate, 存在しなければinset)
     *
     * @throw Exception
     */
    public function upsert(Array $p)
    {
        if ( ! isset($p['FH']) || ! is_array($p['FH'])) {
            return;
        }

        try {
            foreach ($p['FH'] as $type => $label) 
            {
                $age = isset($p[$type.'_age']) ? $p[$type.'_age'] : '';

                // 双子区分ID {0:双子でない 1:一卵性 2:二卵性}
                if (isset($p[$type.'_twins_div_id'])) {
                    $twins_div_id = $p[$type.'_twins_div_id'];
                } else {
                    $twins_div_id = self::isTwins($age, $type) ? 1 : 0;
                }

                $params = array(
                    'card_number'            => $this->card_number, 
                    'type'                   => $type, 
                    'class'                  => self::getKlass($type),
                    'label'                  => $label, 
                    'alive'                  => isset($p[$type.'_alive']) ? $p[$type.'_alive'] : '', 
                    'cohabitation'           => isset($p[$type.'_cohabitation']) ? $p[$type.'_cohabitation'] : '', 
                    'age'                    => $age, 
                    'serious_disease'        => isset($p[$type.'_serious_disease']) ? $p[$type.'_serious_disease'] : '', 
                    'serious_disease_name'   => isset($p[$type.'_serious_disease_name']) ? $p[$type.'_serious_disease_name'] : '', 
                    'serious_disease_y'      => isset($p[$type.'_serious_disease_y']) ? $p[$type.'_serious_disease_y'] : '', 
                    'serious_disease_m'      => isset($p[$type.'_serious_disease_m']) ? $p[$type.'_serious_disease_m'] : '', 
                    'serious_disease_d'      => isset($p[$type.'_serious_disease_d']) ? $p[$type.'_serious_disease_d'] : '', 
                    'serious_disease_status' => isset($p[$type.'_serious_disease_status']) ? $p[$type.'_serious_disease_status'] : '', 
                    'allergy'                => isset($p[$type.'_allergy']) ? $p[$type.'_allergy'] : '', 
                    'allergy_items'          => isset($p[$type.'_allergy_items']) ? join(',', $p[$type.'_allergy_items']) : '', 
                    'allergy_y'              => isset($p[$type.'_allergy_y']) ? $p[$type.'_allergy_y'] : '', 
                    'allergy_m'              => isset($p[$type.'_allergy_m']) ? $p[$type.'_allergy_m'] : '', 
                    'allergy_d'              => isset($p[$type.'_allergy_d']) ? $p[$type.'_allergy_d'] : '', 
                    'allergy_status'         => isset($p[$type.'_allergy_status']) ? $p[$type.'_allergy_status'] : '', 
                    'self_card_number'       => isset($p[$type.'_card_number']) ? $p[$type.'_card_number'] : '', 
                    'gender_div_id'          => self::getGengerDivId($type), 
                    'generation'             => self::getGeneration($type), 
                    'divorce_flg'            => isset($p[$type.'_divorce_flg']) ? $p[$type.'_divorce_flg'] : 0, 
                    'twins_div_id'           => $twins_div_id, 
                    'disease_status_div_id'  => isset($p[$type.'_disease_status_div_id']) ? $p[$type.'_disease_status_div_id'] : 1, // 病気状態区分ID {1:通常状態 2:病気遺伝子あり 3:病気発症}
                    'orderby'                => self::getOrder($type),
                );

                if ($res = $this->dao->fetch_BY_card_number_AND_type($this->card_number, $type)) {
                    // update
                    $family_history_id = $res['family_history_id'];
                    $this->dao->update($family_history_id, $params, $this->now_datetime);
                } else {
                    // insert
                    $this->dao->insert($params, $this->now_datetime);
                }
            }
        } catch (Exception $e) {
            throw new Exception;
        }
    }

    protected static function getOrder($type)
    {
        if ($type === 'F_GF') return 1; // 父方祖父
        if ($type === 'F_GM') return 2; // 父方祖母
        if ($type === 'M_GF') return 3; // 母方祖父
        if ($type === 'M_GM') return 4; // 母方祖母

        if ($type === 'FATHER') return 5; // 父
        if ($type === 'MOTHER') return 6; // 母

        if (preg_match('/^BIG_BRO([0-9]*)$/', $type, $cap)) {
            $num = isset($cap[1]) ? $cap[1] : 0;
            return (20 + $num); // 兄
        }
        if (preg_match('/^BIG_SIS([0-9]*)$/', $type, $cap)) {
            $num = isset($cap[1]) ? $cap[1] : 0;
            return (40 + $num); // 姉
        }
        if (preg_match('/^KID_BRO([0-9]*)$/', $type, $cap)) {
            $num = isset($cap[1]) ? $cap[1] : 0;
            return (60 + $num); // 弟
        }
        if (preg_match('/^KID_SIS([0-9]*)$/', $type, $cap)) {
            $num = isset($cap[1]) ? $cap[1] : 0;
            return (80 + $num); // 妹
        }

        return 999;
    }

    /**
     * 今回の処理で更新されなかったレコードを削除
     *
     * @throw Exception
     */
    public function delete()
    {
        try {
            $this->dao->delete_BY_card_number_NOT_upd_datetime($this->card_number, $this->now_datetime);
        } catch (Exception $e) {
            throw new Exception;
        }
    }



    /*public static function getSiblingsNum(Array $p)
    {
        $big_bro_num = $big_sis_num = $kid_bro_num = $kid_sis_num = 0;
        foreach (array_keys($p['FH']) as $k) {
            switch (true) {
                case (strpos($k, 'BIG_BRO') === 0):
                    ++$big_bro_num;
                    break;

                case (strpos($k, 'BIG_SIS') === 0):
                    ++$big_sis_num;
                    break;

                case (strpos($k, 'KID_BRO') === 0):
                    ++$kid_bro_num;
                    break;

                case (strpos($k, 'KID_SIS') === 0):
                    ++$kid_sis_num;
                    break;
            }
        }

        return array($big_bro_num, $big_sis_num, $kid_bro_num, $kid_sis_num);
    }*/



    protected static function getKlass($type)
    {
        if ($type === 'FATHER') return 'FATHER'; // 父
        if ($type === 'MOTHER') return 'MOTHER'; // 母

        if (preg_match('/_GF$/', $type)) return 'GRANDFATHER'; // 祖父
        if (preg_match('/_GM$/', $type)) return 'GRANDMOTHER'; // 祖母

        if (strpos($type, 'BIG_BRO') === 0) return 'BIG_BRO'; // 兄
        if (strpos($type, 'BIG_SIS') === 0) return 'BIG_SIS'; // 姉
        if (strpos($type, 'KID_BRO') === 0) return 'KID_BRO'; // 弟
        if (strpos($type, 'KID_SIS') === 0) return 'KID_SIS'; // 妹

        return '';
    }

    protected static function getGengerDivId($type)
    {
        $man   = 1;
        $woman = 2;

        if ($type === 'FATHER') return $man;   // 父
        if ($type === 'MOTHER') return $woman; // 母

        if (preg_match('/_GF$/', $type)) return $man;   // 祖父
        if (preg_match('/_GM$/', $type)) return $woman; // 祖母

        if (preg_match('/_BRO[1-9]*$/', $type)) return $man;   // 兄、弟
        if (preg_match('/_SIS[1-9]*$/', $type)) return $woman; // 姉、妹

        return '';
    }

    protected static function getGeneration($type)
    {
        if ($type === 'FATHER' || $type === 'MOTHER') return 2; // 父、母

        if (preg_match('/_GF$/', $type)) return 1; // 祖父
        if (preg_match('/_GM$/', $type)) return 1; // 祖母

        if (preg_match('/_BRO[1-9]*$/', $type)) return 3; // 兄、弟
        if (preg_match('/_SIS[1-9]*$/', $type)) return 3; // 姉、妹

        return '';
    }

    protected static function isTwins($age, $type)
    {
        if ($age === '') return false;
        if (self::getGeneration($type) !== 3) return false;
        if ($age == Answer::get(2, 'age')) return true;

        return false;
    }


}


