<?php
/* vim:se et ts=4 sw=4 sts=4 fdm=marker: */
/**
 * Port Check for Patient on Sanyo System
 * 
 * LICENSE: 
 * 
 * @copyright 2011 Topazos, Inc.
 * @license http://www.topazos.co.jp/license/3_0.txt PHP License 3.0
 * @version CVS: $Id: $
 * @link http://www.topazos.co.jp/package/PackageName
 * @since File available since Release 1.0.0
 */

/**
 *  三洋システム : 患者ポート確認
 * 
 * <要件>
 * - [診察券番号]を元に患者登録情報の有無を確認する
 * - 問診アプリの初期画面[診察券番号]を入力時にソケット通信にてチェックを行う
 * 
 * <仕様>
 * - 三洋システムへはイントラ側からのアクセスのみ可能なので、中間APIを実装する
 *   -> 1.問診アプリ, 2.中間API, 3.三洋システムの3点間通信となる
 * - [1.問診アプリ]からは、診察券番号のみを[2.中間API]へ送信する
 * - [2.中間API]は接続仕様書を元に[3.三洋システム]と通信し必要な情報のみを取得する
 * - [2.中間API]で患者登録情報が正常に取得できた場合に[1.問診アプリ]へ正常コードを
 *   取得できなかった場合、又はエラーが発生した場合はエラーコードを返す
 * 
 * <中間API返信コード>
 * 000 : 存在する(正常)
 * 001 : 存在しない
 * 100 : 通信エラー(中間API接続エラー)
 * 101 : 通信エラー(中間API不正アクセス)
 * 200 : 通信エラー(三洋システム側エラー)
 * 201 : 動作確認エラー(三洋システム側エラー)
 * 999 : システムエラー(三洋システム側エラー)
 * 
 * <通信構成>
 * a. [1.問診アプリ] > [2.中間API]
 *     GETリクエストでHTTP通信で送信(中間サーバのアクセスログに履歴保存が可能)
 * b. [2.中間API] > [3.三洋システム]
 *     ソケット(192.168.100.95:9031)で送信
 * c. [3.三洋システム] > [2.中間API]
 *     ソケットで返信
 * d. [2.中間API] > [1.問診アプリ]
 *     HTTP通信で返信
 * 
 * Sample Code
 * 
 * $obj = new PortCheck;
 * $stat = $obj->isPatient('11111111');
 * if ($stat === FALSE) {
 *     print_r($obj->getError());
 * } else {
 *     echo($obj->getPatientCode() . "\n");
 * }
 * 
 */

/**
 * Port Check Class
 */
class PortCheck
{
    /**
     * Constructor
     */
    public function __construct($isDev = FALSE)
    {
        $host = 'monshin.root-n.com';
        if ($isDev !== TRUE) {
            $host = '本番中間サーバーホスト';
            $host = 'monshin.root-n.com';
        }
        //$host = 'monshin.local.net';
        $this->_url = 'http://' . $host . '/2_api.php';
        //$this->_exam_no = '11111111';
        //$this->_exam_no = '1'; // 存在する
        //$this->_exam_no = '20'; // 存在しない
        //$this->_exam_no = '9999998'; // 存在する
        //$this->_exam_no = $_SERVER['argv'][1]; // 存在する
        $this->_code = '000';
        $this->_msg = '';
    }

    /**
     * Check Patient
     * @return boolean
     */
    public function isPatient($exam_no)
    {
        $this->_exam_no = $exam_no;
        $code = $this->_checkExamNo();
        $result = ($code === '000');
        return $result;
    }

    /**
     * Get Patient Code
     */
    public function getPatientCode()
    {
        return $this->_patient_code;
    }

    /**
     * Get Result
     * @return array
     */
    public function getError()
    {
        $result = array();
        $result['code'] = $this->_code;
        $result['msg']  = $this->_msg;
        return $result;
    }

    /**
     * Get Request Url
     * @return string Url
     */
    private function _getRequestUrl()
    {
        $url = $this->_url;
        $url .= sprintf("?exam_no=%s", $this->_exam_no);
        return $url;
    }

    /**
     * Put Error
     * @param string $code : Error Code
     * @param string $msg  : Error Message
     * @return void
     */
    private function _putError($code, $msg)
    {
        $this->_code = $code;
        $this->_msg = $msg;
    }

    /**
     * Check Exam No
     * @return boolean
     */
    private function _checkExamNo()
    {
        $result = FALSE;
        $info = parse_url($this->_getRequestUrl());
        $fp = fsockopen($info['host'], 80, $errno, $errstr, 10);
        if (!$fp) {
            $msg = "中間APIへの接続に失敗しました<$errno:$errstr>";
            $this->_putError('100', $msg);
        } else {
            $out = "GET {$info['path']}?{$info['query']} HTTP/1.1\r\n";
            $out .= "Host: {$info['host']}\r\n";
            $out .= "Connection: Close\r\n\r\n";
            fwrite($fp, $out);
            $body = FALSE;
            $code = $msg = NULL;
            while (($line = fgets($fp, 1024)) !== FALSE) {
                $line = trim($line);
                if (empty($line)) {
                    $body = TRUE;
                    continue;
                }
                if ($body !== FALSE) {
                    list($code, $msg) = explode(':', $line);
                    break;
                }
            }
            fclose($fp);
            switch ($code) {
                case '000' : // 存在する
                    $result = $code;
                    $this->_patient_code = $msg;
                    break;
                case '001' : // 存在しない
                    $result = $code;
                    $this->_putError($code, $msg);
                    break;
                case '101' : // APIエラー
                case '200' :
                case '201' :
                case '999' :
                    $this->_putError($code, $msg);
                    break;
                default : // その他接続エラー
                    $code = '100';
                    $msg = "中間APIの結果取得に失敗しました<$code>";
                    $this->_putError($code, $msg);
            }
        }
        return $result;
    }
}

?>
