<?php
/* vim:se et ts=4 sw=4 sts=4 fdm=marker: */
/**
 * Create Xml for Recipe
 * 
 * LICENSE: 
 * 
 * @copyright 2011 Topazos, Inc.
 * @license http://www.topazos.co.jp/license/3_0.txt PHP License 3.0
 * @version CVS: $Id: $
 * @link http://www.topazos.co.jp/package/PackageName
 * @since File available since Release 1.0.0
 */

require_once dirname(__FILE__) . '/XmlCommon.php';

/**
 * Sample Code
 *
 * $obj = new XmlRecipe('Shift_JIS');
 * // Sample Patient Info
 * $patient = $obj->parsePatientInfo('123456');
 * // Sample Basic Info
 * $basic = $obj->parseBasicInfo(
 *     '123456890',
 *     '入院',
 *     '20110929120000',
 *     '山田　太郎',
 *     'user1',
 *     '初診',
 *     '内科',
 *     '\\\\',
 *     '国保',
 *     '001',
 *     '院外',
 *     'ネプライザーの種類をまちがえたので修正'
 * );
 * // Sample Category
 * $units_a = $obj->parseUnits('999999999', '', '診察', 'user_a', '医師 A', '20111027000000', '20111027000000',
 *                                 array(
 *                                     $obj->parseOrder('1216', '特定疾患', '1', '', ''),
 *                                     $obj->parseOrder('9999', 'ネフローゼ', '9', 'bit', '3'),
 *                                 ),
 *                                 $obj->parseAdmin('', '', '1', '回'),
 *                                 'メモA'
 *                             );
 * $units_b = $obj->parseUnits('888888888', '', '診察', 'user_b', '医師 B', '20111027000000', '20111027000000',
 *                                 array(
 *                                     $obj->parseOrder('8888', '風邪', '2', 'km', '8'),
 *                                 ),
 *                                 $obj->parseAdmin('', 'うさ', '10', '台'),
 *                                 'メモB'
 *                             );
 * $obj->setPatientInfo($patient);
 * $obj->setBasicInfo($basic);
 * $obj->setCategory($obj->parseCategory('診察', array($units_a, $units_b)));
 * $obj->setCategory($obj->parseCategory('投薬', array($units_b)));
 * $xml = $obj->execute();
 * $obj->save($xml, '2011-10-27 00:00:00'); // 第3引数は'dev'or'prod'を指定(デフォルト: dev)
 * 
 */



/**
 * Xml Recipe Class
 */
class XmlRecipe extends XmlCommon
{

    /**
     * Variables
     */
    private $_fileprefix  = 'RENKEI001'; // 事前に取決めた値(固定)
    private $_systeminfo  = 'ABCDEFG'; // 事前に取決めた値(固定)
    private $_patientinfo = '';        // 患者コード(患者当初送信で取得)
    private $_basicinfo   = array();   // 基本情報
    private $_category    = array();   // カテゴリ情報(処方内容)

    /**
     * Constructor
     */
    public function __construct($enc = 'UTF-8')
    {
        parent::__construct($enc);
    }

    /**
     * Execute
     */
    public function execute()
    {
        $data = array();
        // SystemInfo...
        $data['SystemInfo'] = $this->_getSystemInfo();
        // Patient_Info...
        $data['Consultation_Data'][0]['Patient_Info'] = $this->_getPatientInfo();
        // Basic_Info...
        $data['Consultation_Data'][0]['Consultation'][0]['Basic_Info'] = $this->_getBasic_Info();
        // Order_Info(Category)...
        foreach ($this->_category as $val) {
            $data['Consultation_Data'][0]['Consultation'][0]['Order_Info'][0][]['Order_Category'] = $this->_getOrder_Category($val);
        }
        $data = $this->_removeSjisGaiji($data);
        $el = $this->_xmlElement('Consultation_Datas', $data, array('S_Version' => '0200'));
        $this->x->appendChild($el);
        $result = $this->x->saveXML();
        if ($this->_enc != 'UTF-8') {
            $result = $this->_outEncode($result, $this->_enc);
        }
        return $result;
    }

    // {{{ Parser...
    public function parsePatientInfo($patient_id)
    {
        return array('Patient_ID' => $patient_id);
    }

    public function parseBasicInfo($data_id, $flag, $date, $doctor, $doctor_id, $type, $depart, $depart_id, $insurance, $insurance_id, $inout, $message)
    {
        $basic = array(
            'Consultation_DataID' => $data_id,
            'InPatientFlag' => $flag,
            'Consultation_Date' => $date,
            'Consultation_Doctor' => $doctor,
            'DoctorID' => $doctor_id,
            'Consultation_Type' => $type,
            'Consultation_Department' => $depart,
            'DepartmentID' => $depart_id,
            'Insurance' => $insurance,
            'InsuranceID' => $insurance_id,
            'Prescription_INOUT' => $inout,
            'Message' => $message,
        );
        return $basic;
    }

    public function parseUnits($units_id, $type, $appli, $code, $name, $time, $last, $order, $admin, $memo)
    {
        $units = array(
                    'Order_UnitsID' => $units_id,
                    'Order_type' => $type,
                    'Application' => $appli,
                    'InputUserCode' => $code,
                    'InputUserName' => $name,
                    'InputTime' => $time,
                    'LastUpdateTime' => $last,
                    'Order' => $order,
                    'Order_Administration' => $admin, 
                    'OrderUnits_Memo' => $memo,
                );
        return $units;
    }

    public function parseOrder($code = '', $name = '', $count = '', $unit = '', $cutoff = '')
    {
        $order = array(
                    'Code' => $code,
                    'Name' => $name,
                    'Count' => $count,
                    'Unit' => $unit,
                    'Cutoff' => $cutoff,
                    // 'SeqNo' => $seqno, // Comment out by Auto Count Up...
                );
        return $order;
    }

    public function parseAdmin($code = '', $usage = '', $count = '', $unit = '')
    {
        $admin = array(
                    'Code' => $code,
                    'Usage' => $usage,
                    'Count' => $count,
                    'Unit' => $unit,
                );
        return $admin;
    }

    public function parseCategory($category, $units)
    {
        $category = array(
                        'Category' => $category,
                        'Order_Units' => $units,
                    );
        return $category;
    }
    // }}}

    // {{{ Setter...
    /**
     * Set Patient Code
     */
    public function setPatientInfo($code)
    {
        $this->_patientinfo = $code;
    }

    /**
     * Set Basic Info
     */
    public function setBasicInfo($data)
    {
        $this->_basicinfo = $data;
    }

    /**
     * Set Category
     */
    public function setCategory($data)
    {
        $this->_category[] = $data;
    }
    // }}}

    // {{{ Node Tools...
    /**
     * Get SystemInfo
     */
    private function _getSystemInfo()
    {
        $data[0] = array('SystemName' => array($this->_systeminfo));
        return  $data;
    }

    /**
     * Get Patient_Info
     */
    private function _getPatientInfo()
    {
        $patient = $this->_patientinfo;
        $data[0] = array('Patient_ID' => array($patient['Patient_ID']));
        return  $data;
    }

    /**
     * Get Basic_Info
     */
    private function _getBasic_Info()
    {
        $basic = $this->_basicinfo;
        $data[0]['InPatientFlag']           = array($basic['InPatientFlag']);
        $data[0]['Consultation_Date']       = array($basic['Consultation_Date']);
        $data[0]['Consultation_Doctor']     = array($basic['Consultation_Doctor'], array('DoctorID' => $basic['Consultation_Doctor']));
        $data[0]['Consultation_Type']       = array($basic['Consultation_Type']);
        $data[0]['Consultation_Department'] = array($basic['Consultation_Department'], array('DepartmentID' => $basic['DepartmentID']));
        $data[0]['Insurance']               = array($basic['Insurance'], array('InsuranceID' => $basic['InsuranceID']));
        $data[0]['Prescription_INOUT']      = array($basic['Prescription_INOUT']);
        $data[0]['Message']                 = array($basic['Message']);
        $data[1] = array('Consultation_DataID' => $basic['Consultation_DataID']);
        return $data;
    }

    /**
     * Get Order_Category
     */
    private function _getOrder_Category($category)
    {
        foreach ($category['Order_Units'] as $val) {
            $data[0][]['Order_Units'] = $this->_getOrder_Units($val);
        }
        $data[1] = array('Category' => $category['Category']);
        return $data;
    }

    /**
     * Get Order_Units
     */
    private function _getOrder_Units($unit)
    {
        $admin = $unit['Order_Administration'];
        $memo  = $unit['OrderUnits_Memo'];
        $order = $unit['Order'];
        $seq_no = 1; // Auto Count Up...
        foreach ($order as $val) {
            $val['SeqNo'] = $seq_no++;
            $data[0][] = array('Order' => $this->_getOrder($val));
        }
        $data[0][] = array('Order_Administration' => array('', array(
                                                            'Code'  => $admin['Code'],
                                                            'Usage' => $admin['Usage'],
                                                            'Count' => $admin['Count'],
                                                            'Unit'  => $admin['Unit'],
                                                        )));
        $data[0][] = array('OrderUnits_Memo' => array($memo));
        $data[1] = array(
                        'Order_UnitsID'  => $unit['Order_UnitsID'],
                        'Order_type'     => $unit['Order_type'],
                        'Application'    => $unit['Application'],
                        'InputUserCode'  => $unit['InputUserCode'],
                        'InputUserName'  => $unit['InputUserName'],
                        'InputTime'      => $unit['InputTime'],
                        'LastUpdateTime' => $unit['LastUpdateTime'],
                        );
        return $data;
    }

    /**
     * Get Order
     */
    private function _getOrder($order)
    {
        $data[0] = '';
        $data[1] = array(
                        'Code'   => $order['Code'],
                        'Name'   => $order['Name'],
                        'Count'  => $order['Count'],
                        'Unit'   => $order['Unit'],
                        'Cutoff' => $order['Cutoff'],
                        'SeqNo'  => $order['SeqNo'],
                        );
        return $data;
    }
    // }}}

    // {{{ Save Tools...
    private function _parseSaveTime($date)
    {
        sscanf($date, "%04d-%02d-%02d %02d:%02d:%02d", $y,$m,$d,$h,$i,$s);
        $result = sprintf("%04d%02d%02d%02d%02d%02d", $y,$m,$d,$h,$i,$s);
        return $result;
    }

    public function save($data, $mondate, $env = 'dev')
    {
        $base = '/var/www/' . $env . '/xml/recipe';
        $file = $this->getFileName($mondate);
        $path = $base . '/' . $file;
        $i = 1;
        while (is_file($path)) {
            // 念の為、ファイル有無を確認して連番対応...
            $prev = sprintf("_%04d.xml", $i);
            $next = sprintf("_%04d.xml", ++$i);
            $path = str_replace($prev, $next, $path);
        }
        if (($fp = fopen($path, 'w')) !== FALSE) {
            if (flock($fp, LOCK_EX)) {
                fwrite($fp, $data);
                flock($fp, LOCK_UN);
            }
            fclose($fp);
            chmod($path, 0666);
        }
    }

    public function getFileName($mondate)
    {
        $outdate = date('Y-m-d H:i:s');
        $patient_code = $this->_patientinfo['Patient_ID'];
        $file = $this->_fileprefix;                       // 機種識別文字列(固定)事前に決定
        $file .= '_' . (string)$patient_code;             // 患者コード
        $file .= '_' . $this->_parseSaveTime($mondate);   // 受診年月日時分秒
        $file .= '_' . $this->_parseSaveTime($outdate);   // 出力年月日時分秒
        $file .= '_' . '0001.xml';                        // 連番(重複の可能性が無いので固定)
        return $file;
    }
    // }}}

}

?>
