<?php
/**
 * PDO_MySQL Class
 *
 * @package Lambda/lib/DB/PDO
 * @author  rooth
 * @version 0.0.1
 *
 * PHP versions 5
 */

class PDO_MySQL
{
    const PERSISTENT = true;

    protected $pdo;
    protected $host;
    protected $db;
    protected $errlog_dir;

    public function __construct ($host, $db, $user, $pass, $encoding = '')
    {
        $this->host = $host;
        $this->db   = $db;
        $this->errlog_dir = defined('DB_ERRLOG_DIR') ? rtrim(DB_ERRLOG_DIR, '/') : '';
        $options = array(
            PDO::MYSQL_ATTR_SSL_CA => '/var/www/html/er_demo2_8/DigiCertGlobalRootCA.crt.pem',
            PDO::ATTR_PERSISTENT => self::PERSISTENT
        );
        try {
            $this->pdo = new PDO("mysql:host=$host; dbname=$db", $user, $pass, $options);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            if ($encoding) 
		$this->pdo->query("SET NAMES $encoding ; ");
            //if ($encoding) mysql_set_charset($encoding); // MySQL 5.0.7 以降でないと使用できません。
        } catch (PDOException $e) {
            $this->trigger_error($e);
        }
    }

    public function begin()
    {
        return $this->pdo->beginTransaction();
    }

    public function commit()
    {
        return $this->pdo->commit();
    }

    public function rollback()
    {
        return $this->pdo->rollBack();
    }

    public function lastInsertId($name = NULL)
    {
        return $this->pdo->lastInsertId($name);
    }

    public function getPDO () {
        return $this->pdo;
    }

    public function fetchColumn($sql, $array = array())
    {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($array);
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql, $array);
        }
    }

    public function fetch($sql, $array = array())
    {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($array);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql, $array);
        }
    }

    public function fetchAll($sql, $array = array())
    {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($array);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql, $array);
        }
    }

    protected function fetchAllIn()
    {
        /**
         * DAO で、値をキャストして埋め込むことを推奨
         * 
         * public function fetchAllByIds($ids)
         * {
         *    foreach ($ids as $k => $v) {
         *        $ids[$k] = (int)$v;
         *    }
         *    $member_ids = implode(', ', $ids);
         *    $sql = "select * from member where member_id in ({$member_ids}) order by member_id ";
         *    $params = array();
         *    return $this->db->fetchAll($sql, $params);
         * }
         */
    }

    public function execute($sql, $array = array())
    {
        try {
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($array);
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql, $array);
        }
    }

    /**
     * 更新系クエリを実行後、作用した行数を返す
     *
     * @return integer 作用した行数 / 更新に失敗した時: false
     */
    protected function executeWithAffectedRows($sql, $array)
    {
        try {
            $stmt = $this->pdo->prepare($sql);
            $ret = $stmt->execute($array);
            if ($ret) {
                // PDOStatement::rowCount() は DELETE, INSERT, UPDATE 文によって作用した行数を返す。
                // ただし、UPDATE において、変更前と後で何の変更もなかった時(同じ値を更新した時)は
                // 該当する行をカウントしない。
                return $stmt->rowCount();
            } else {
                return false;
            }
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql, $array);
        }
    }

    /**
     * for LOCK TABLES / UNLOCK TABLES
     */
    public function exec($sql)
    {
        try {
            return $this->pdo->exec($sql);
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql);
        }
    }

    /**
     *<pre>
     *
     * FOUND_ROWS()
     * 
     * SELECT ステートメントに LIMIT  節を組み込むことによって、サーバがクライアントに返す
     * レコード数を制限できる。
     * 状況によっては、LIMIT を指定しなかった場合にいくつのレコードが返されるかを、
     * ステートメントを再度実行することなく確認したいことがある。
     * このレコード数を確認するには SELECT ステートメントに SQL_CALC_FOUND_ROWS オプションを指定し、
     * その後 FOUND_ROWS() を呼び出す。
     * 
     * mysql> SELECT SQL_CALC_FOUND_ROWS * FROM tbl_name
     *        WHERE id > 100 LIMIT 10;
     * mysql> SELECT FOUND_ROWS();
     *
     *</pre>
     */
    public function foundRows()
    {
        try {
            $sql = "SELECT FOUND_ROWS();";
            $stmt = $this->pdo->query($sql);
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            $this->trigger_error($e, $sql);
        }
    }

    private function setAttribute($attribute, $value)
    {
        $this->pdo->setAttribute($attribute, $value);
    }

    private function getAttribute($attribute)
    {
        return $this->pdo->getAttribute($attribute);
    }

    protected function trigger_error($e, $sql = null, $arr = null)
    {
        $mes = $e->getMessage()."\n";
        $mes .= "  DB   : {$this->host}:{$this->db}\n";
        if (! is_null($sql)) {
            $mes .= "  SQL  : {$sql}\n";
        }
        if (! is_null($arr)) {
            $mes .= "  Param: ".preg_replace('/\s*\n\s*/', '', var_export($arr, true))."\n";
        }

        if ($this->errlog_dir !== '') {
            if ( ! is_writable($this->errlog_dir)) {
                echo('['.__CLASS__."] Warning: Not found (or Permission denied) ErrorLog Dir: {$this->errlog_dir}\n");
            } else {
                list($Ymd, $His) = explode(' ', date('Y-m-d H:i:s'));
                $log_msg  = "[{$Ymd} {$His}] {$mes}\n";
                $log_file = $this->errlog_dir."/{$Ymd}.err";
                error_log($log_msg, 3, $log_file);
            }
        }

        if (ini_get('display_errors')) {
            echo '<pre>';
            trigger_error($mes, E_USER_ERROR);
            exit;
        }

        // throw PDOException
        throw new PDOException($mes);
    }



} //-- End of class


