<?php
/**
 * バリデーター class
 *
 * @package Lambda/lib
 * @author  rooth
 * @version 0.0.1
 *
 * PHP version 5
 */

class Validator
{
    /**
     * 日付
     * @param str $date  YYYY/MM/DD 形式の日付
     * @param str $delim 区切り文字
     * @return bool
     */
    public static function isDate($date, $delim = '/')
    {
        $arr = explode($delim, $date);

        if (count($arr) !== 3) return false;

        if ( ! preg_match('/^\d{4}$/', $arr[0])) return false;
        if ( ! preg_match('/^\d{2}$/', $arr[1])) return false;
        if ( ! preg_match('/^\d{2}$/', $arr[2])) return false;

        return checkdate($arr[1], $arr[2], $arr[0]);
    }

    /**
     * メールアドレス
     * @param str $email メールアドレス
     * @return bool
     */
    public static function isEmail($email)
    {
        /**
         * [\x20-\x7e] <- 半角英数,記号
         * [\x20] <- 半角スペース
         * [\x40] <- @
         * [\x21-\x3f|\x41-\x7e] <- 半角スペースと @ 以外の半角英数,記号
         */
        $regex = "/^[\x21-\x3f|\x41-\x7e]+" . '@' . "[\x21-\x3f|\x41-\x7e]+" . "\.[0-9A-Za-z]+$/";
        return (bool)preg_match($regex, $email);
    }

    /**
     * カンマ区切りメールアドレス
     * @param str $email カンマ区切りメールアドレス
     * @return bool
     */
    public static function isMultiEmail($email)
    {
        $mails = explode(',', $email);
        foreach ($mails as $m) {
            $m = trim($m);
            if ( ! self::isEmail($m)) return false;
        }
        return true;
    }

    /**
     * URL
     * @param str $url URL
     * @return bool
     */
    public static function isURL($url)
    {
        return (bool)preg_match('|^https?://[-_.!~*\'()a-zA-Z0-9;\/?:\@&=+\$,%#]+$|i', $url); 
    }

    /**
     * 郵便番号
     * @param str $zip 郵便番号
     * @return bool
     */
    public static function isZip($zip)
    {
        if (preg_match('/^\d{3}-\d{4}$/', $zip)) return true;
        if (preg_match('/^\d{7}$/', $zip)) return true;
        return false;
    }

    /**
     * 電話番号
     * @param str $phone 電話番号
     * @return bool
     */
    public static function isPhone($phone)
    {
        return (bool)preg_match('/^[\d-]+$/', $phone);
    }



    /**
     * 正の数？
     */
    public static function isPositiveNumber($v)
    {
        return (bool)preg_match('/^[1-9]\d*$/', $v);
    }

    /**
     * 半角スペースを除く、半角英数と記号 の判定
     */
    public static function isPrintable($v)
    {
        /**
         * [\x21-\x7e] <- 半角スペースを除く、半角英数と記号
         */
        $regex = "/^[\x21-\x7e]+$/";
        return (bool)preg_match($regex, $v);
    }

    /**
     * サブネットマスク形式のIPアドレスに含まれるか検証する
     *
     * @param str   $test_ip    テストするIPアドレス
     * @param array $permit_ips IPアドレスのリスト
     * @return bool
     *
     * [usage]
     * $permit_ips = array(
     *     '210.153.84.0/24', 
     *     '210.136.161.0/24', 
     * );
     * $test_ip = '210.161.123.12';
     * echo isPermittedIP($test_ip, $permit_ips) ? 'OK' : 'NG';
     * 
     */
    public static function isPermittedIP($test_ip, Array $permit_ips)
    {
        foreach ($permit_ips as $permit_ip) {
            list($ip, $bit_mask) = explode('/', $permit_ip);
            $ip_long = ip2long($ip) >> (32 - $bit_mask);
            $test_ip_long = ip2long($test_ip) >> (32 - $bit_mask);
            if ($ip_long === $test_ip_long) {
                return true;
            }
        }

        return false;
    }



}


