<?php
/* vim:se et ts=4 sw=4 sts=4: */
/**
 * Patient Checker Api for Patient on Sanyo System
 * 
 * LICENSE: 
 * 
 * @copyright 2011 Topazos, Inc.
 * @license http://www.topazos.co.jp/license/3_0.txt PHP License 3.0
 * @version CVS: $Id: $
 * @link http://www.topazos.co.jp/package/PackageName
 * @since File available since Release 1.0.0
 */

/**
 *  三洋システム : 患者ポート確認API
 * 
 * <要件>
 * - [診察券番号]を元に患者登録情報の有無を確認する
 * - 問診アプリの初期画面[診察券番号]を入力時にソケット通信にてチェックを行う
 * 
 * <仕様>
 * - 三洋システムへはイントラ側からのアクセスのみ可能なので、中間APIを実装する
 *   -> 1.問診アプリ, 2.中間API, 3.三洋システムの3点間通信となる
 * - [1.問診アプリ]からは、診察券番号のみを[2.中間API]へ送信する
 * - [2.中間API]は接続仕様書を元に[3.三洋システム]と通信し必要な情報のみを取得する
 * - [2.中間API]で患者登録情報が正常に取得できた場合に[1.問診アプリ]へ正常コードを
 *   取得できなかった場合、又はエラーが発生した場合はエラーコードを返す
 * 
 * <中間API返信コード>
 * 000 : 存在する(正常)
 * 001 : 存在しない
 * 100 : 通信エラー(中間API接続エラー) ※このAPIに接続ができない場合
 * 101 : 通信エラー(中間API不正アクセス)
 * 200 : 通信エラー(三洋システム側エラー)
 * 201 : 動作確認エラー(三洋システム側エラー)
 * 999 : システムエラー(三洋システム側エラー)
 * 
 * <通信構成>
 * a. [1.問診アプリ] > [2.中間API]
 *     GETリクエストでHTTP通信で送信(中間サーバのアクセスログに履歴保存が可能)
 * b. [2.中間API] > [3.三洋システム]
 *     ソケット(192.168.100.95:9031)で送信
 * c. [3.三洋システム] > [2.中間API]
 *     ソケットで返信
 * d. [2.中間API] > [1.問診アプリ]
 *     HTTP通信で返信
 * 
 * <処理結果ログ>
 * 保存場所 : api_log/yyyy/yyyymmdd.log
 * 本APIと同階層に[api_log](777)で作成する必要がある
 * 
 * <必須設定項目> 
 * 三洋システムサーバーIPアドレス
 *     const host = '192.168.100.95';
 * 改行コード定数の値を空文字にする
 *     const c_lf  = ''; //"\x0A";
 * 
 */


/**
 * Patient Checker
 */
class PatientChecker
{

    /**
     * Constant.
     */
    //const host = 'monshin.root-n.com';
    //const host = '127.0.0.1';
    const host = '192.168.100.95';
    const port = 9031;
    //const c_stx = chr(2); //"\x02";
    //const c_etx = chr(3); //"\x03";
    //const c_ack = chr(6); //"\x06";
    //const c_nak = chr(15); //"\x15";
    //const c_syn = chr(16); //"\x16";
    //const c_dle = chr(10); //"\x10";
    //const c_eot = chr(4); //"\x04";
    //const c_enq = chr(5); //"\x05";
    //const c_lf  = ''; //"\x0A";
    const fenc = 'SJIS';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->_logdir = 'api_log';
        $this->_data = array();
	    $this->c_stx = chr(2); //"\x02";
	    $this->c_etx = chr(3); //"\x03";
	    $this->c_ack = chr(6); //"\x06";
	    //$this->c_nak = chr(15); //"\x15";
	    $this->c_nak = chr(21); //"\x15";
	    //$this->c_syn = chr(16); //"\x16";
	    $this->c_syn = chr(22); //"\x16";
	    //$this->c_dle = chr(10); //"\x10";
	    $this->c_dle = chr(16); //"\x10";
	    $this->c_eot = chr(4); //"\x04";
	    $this->c_enq = chr(5); //"\x05";
	    $this->c_lf  = ''; //"\x0A";
	    //$this->c_lf  = "\x0A"; //"\x0A"; // SANYOモックテスト(3_sanyo.php)ではLFが必要
    }

    // {{{ Common Operation...
    /**
     * Prepare
     */
    public function prepare()
    {
        $this->_exam_no = (string)$_GET['exam_no'];
    }

    /**
     * Validate
     */
    public function validate()
    {
        $result = strlen((string)$this->_exam_no) != 0;
        if ($result === FALSE) {
            $this->_rescode = '101';
        } else {
            $result = $this->_sendConfirmPacket($code, $err);
            if ($result === FALSE) {
                $msg = "三洋通信エラー($code) $err";
                $this->_rescode = $code;
                $this->_putError($code, $msg);
            }
        }
        return $result;
    }

    /**
     * Execute
     */
    public function execute()
    {
        $message = $this->_getRequestMessage($this->_exam_no);
        $stat = $this->_sendRequestMessage($message, $code, $err);
        if ($stat === TRUE) {
            $rescode = '000';
        } else {
            switch ($code) {
                case '001' :
                    $msg = "患者データが存在しません($code) $err";
                    $rescode = '001';
                    break;
                case '999' :
                    $msg = "三洋システムエラー($code) $err";
                    $rescode = '999';
                    break;
                default :
                    $msg = "三洋通信エラー($code) $err";
                    $rescode = '200';
            }
            $this->_putError($code, $msg);
        }
        $this->_rescode = $rescode;
    }

    /**
     * Complete
     */
    public function complete()
    {
        if ($this->_rescode === '000') {
            $msg = $this->getPatientCode();
            $msg .= '&' . $this->getInsuranceNo();
            $msg .= '&' . $this->getInsuranceName();
        } else {
            $msg = $this->getError();
        }
        $result = sprintf("%s:%s", $this->_rescode, $msg);
        echo($result);
        $this->logging($result);
    }
    // }}}

    // {{{ Log Tools...
    public function logging($str)
    {
        $log = sprintf("[%s] {%s} %s\n", date('Y-m-d H:i:s'), $this->_exam_no, $str);
        $ydir = $this->_logdir . '/' . date('Y');
        if (! is_dir($ydir)) {
            mkdir($ydir);
            chmod($ydir, 0777);
        }
        $file = $ydir . '/' . date('Ymd') . '.log';
        error_log($log, 3, $file);
    }
    // }}}

    // {{{ Gettter / Setter...
    /**
     * Get Result Code
     */
    public function getResultCode()
    {
        return $this->_rescode;
    }

    /**
     * Get Patient Code, if exists
     */
    public function getPatientCode()
    {
        if (isset($this->_data['d9'])) {
            return $this->_data['d9'];
        }
    }

    /**
     * Get Insurance No, if exists
     */
    public function getInsuranceNo()
    {
        if (isset($this->_data['d12'])) {
            return $this->_data['d12'];
        }
    }

    /**
     * Get Insurance Name, if exists
     */
    public function getInsuranceName()
    {
        if (isset($this->_data['d51'])) {
            $name = substr($this->_data['d51'], 2);
            $name = substr($name, 0, -2);
            return $name;
        }
    }

    /**
     * Get Error
     */
    public function getError()
    {
        return $this->_msg;
    }

    /**
     * Put Error
     * @param string $code : Error Code
     * @param string $msg  : Error Message
     * @return void
     */
    private function _putError($code, $msg)
    {
        $this->_code = $code;
        $this->_msg = $msg;
    }

    /**
     * Get Data
     */
    public function getData()
    {
        return $this->_data;
    }
    // }}}

    // {{{ Socket Tools...
    /**
     * Send Confirm Packet
     */
    private function _sendConfirmPacket(&$code, &$err)
    {
        $result = FALSE;
        $fp = fsockopen(self::host, self::port, $errno, $errstr, 30);
        if (! $fp) {
            $err = "三洋システムへの接続に失敗しました[確認]($errno:$errstr)";
            $code = 201;
        } else {
            //fwrite($fp, self::c_enq . self::c_lf); // Send ENQ
            fwrite($fp, $this->c_enq . $this->c_lf); // Send ENQ
            $ret = fread($fp, 1);                  // Get ACK
            //$result = ($ret === self::c_ack);
            $result = ($ret === $this->c_ack);
            fclose($fp);
        }
        return $result;
    }

    /**
     * Send Request Message
     */
    private function _sendRequestMessage($message, &$code, &$err)
    {
        $result = FALSE;
        $code = NULL;
        $fp = fsockopen(self::host, self::port, $errno, $errstr, 30);
        if (! $fp) {
            $err = "三洋システムへの接続に失敗しました[要求]($errno:$errstr)";
            $code = '200';
        } else {
            //$cnt1 = $cnt2 = 5;
            //do {
                //fwrite($fp, $message . self::c_lf);
                fwrite($fp, $message . $this->c_lf);
                $ret = fread($fp, 1082); // 最長1082バイト
                $result = $this->_parse($ret, $code);
                if (! is_null($result)) {
                    // NAKで無ければACKを送信...
                    //do {
                        //fwrite($fp, self::c_ack . self::c_lf); // Send ACK
                        fwrite($fp, $this->c_ack . $this->c_lf); // Send ACK
                        //$ret = fread($fp, 1);                  // Get ACK
                        //if ($ret === self::c_ack) {
                        //if ($ret === $this->c_ack) {
                        //    // ACKの場合...
                        //    break;
                        //}
                    //} while ($cnt2--);
                    //break;
                } else {
                        fwrite($fp, $this->c_nak . $this->c_lf); // Send NAK
                    //break;
		}
            //} while ($cnt1--);
            fclose($fp);
        }
        return $result;
    }

    /**
     * Parse
     */
    private function _parse($data, &$code)
    {
        //return $this->_parseAA1($data, $code);
        return $this->_parseA11($data, $code);
    }

    private function _parseAA1($data, &$code)
    {
        $code = NULL;
        $d1 = substr($data, 0, 1);   // 1 STX
        //if ($d1 === self::c_nak) { // NAK
        if ($d1 == $this->c_nak) { // NAK
            $result = NULL;
        //} elseif ($d1 === self::c_stx) { // STX
        } elseif ($d1 == $this->c_stx) { // STX
            $d2 = substr($data, 1, 2);   // 2 拡張部
            $d3 = substr($data, 3, 4);   // 4 電文区分
            $d4 = substr($data, 7, 3);   // 3 ブロック区分
            $d5 = substr($data, 10, 1);  // 1 予備
            $d6 = substr($data, 11, 3);  // 3 データ区分
            $d7 = substr($data, 14, 1);  // 1 サブ区分
            $d8 = substr($data, 15, 1);  // 1 情報種別
            $d9 = substr($data, 16, 13); // 13 患者コード
            $d10 = substr($data, 29, 3); // 3 主科
            $d11 = substr($data, 32, 1); // 1 外/入
            $d12 = substr($data, 33, 3); // 3 保険組No
            $d13 = substr($data, 36, 2); // 2 RSV
            $d14 = substr($data, 38, 1); // 1 年齢区分
            $d15 = substr($data, 39, 1); // 1 所得区分
            $d16 = substr($data, 40, 1); // 1 認定区分
            $d17 = substr($data, 41, 1); // 1 医療/介護フラグ
            $d18 = substr($data, 42, 13); // 13 RSV
            $d19 = substr($data, 55, 18); // 18 患者ID
            $d20 = substr($data, 73, 6); // 6 最終来院月
            $d21 = substr($data, 79, 40); // 40 カナ氏名
            $d22 = substr($data, 119, 44); // 44 漢字氏名
            $d23 = substr($data, 163, 1); // 1 性別
            $d24 = substr($data, 164, 8); // 8 漢字元号
            $d25 = substr($data, 172, 8); // 8 生年月日
            $d26 = substr($data, 180, 1); // 1 ETX
            $result = TRUE;
        //} elseif ($d1 === self::c_dle) { // DLE
        } elseif ($d1 == $this->c_dle) { // DLE
            $d2 = substr($data, 1, 3);   // 3 識別子
            $tmp = substr($data, 4); // 残り部分(エラーコード+EOT)
            $len = strlen($tmp);
            $d3 = ''; // 5 エラーコード(最大5バイト)
            for ($i = 0; $i < $len; $i++) {
                //if ($tmp[$i] === self::c_eot) { break; } // EOTになったら終了
                if ($tmp[$i] === $this->c_eot) { break; } // EOTになったら終了
                $d3 .= $tmp[$i];
            }
            //$d4 = self::c_eot; // 1 EOT
            $d4 = $this->c_eot; // 1 EOT
            $code = $d3;
            $result = FALSE;
        } else {
            $result = FALSE;
        }
        $cnt = 26;
        for ($i = 1; $i <= $cnt; $i++) {
            $var = 'd' . $i;
            if (isset($$var)) {
                if ($i != 1 && $i != 26) {
                    $$var = mb_convert_encoding($$var, 'UTF-8', self::fenc);
                }
                $this->_data[$var] = $$var;
            }
        }
        return $result;
    }

    private function _parseA11($data, &$code)
    {
        $code = NULL;
        $d1 = substr($data, 0, 1);   // 1 STX
        if ($d1 == $this->c_nak) {   // NAK
            $result = NULL;
        } elseif ($d1 == $this->c_stx) { // STX
            $d2 = substr($data, 1, 2);   // 2 拡張部
            $d3 = substr($data, 3, 4);   // 4 電文区分
            $d4 = substr($data, 7, 3);   // 3 ブロック区分
            $d5 = substr($data, 10, 1);  // 1 予備
            $d6 = substr($data, 11, 3);  // 3 データ区分
            $d7 = substr($data, 14, 1);  // 1 サブ区分
            $d8 = substr($data, 15, 1);  // 1 情報種別
            $d9 = substr($data, 16, 10); // 10 患者コード
            $d10 = substr($data, 26, 2); // 2 主科
            $d11 = substr($data, 28, 1); // 1 外/入
            $d12 = substr($data, 29, 3); // 3 保険組No
            $d13 = substr($data, 32, 2); // 2 RSV
            $d14 = substr($data, 34, 1); // 1 年齢区分
            $d15 = substr($data, 35, 1); // 1 所得区分
            $d16 = substr($data, 36, 1); // 1 認定区分
            $d17 = substr($data, 37, 1); // 1 医療/介護フラグ
            $d18 = substr($data, 38, 10);// 10 患者コード
            $d19 = substr($data, 48, 6); // 6 患者ID
            $d20 = substr($data, 54, 6); // 6 最終来院月
            $d21 = substr($data, 60, 16); // 40 カナ氏名
            $d22 = substr($data, 76, 10); // 10 RSV
            $d23 = substr($data, 86, 20); // 20 漢字氏名
            $d24 = substr($data, 106, 1); // 1 性別
            $d25 = substr($data, 107, 8); // 8 漢字元号
            $d26 = substr($data, 115, 8); // 8 生年月日
            $d27 = substr($data, 123, 1); // 1 RSV
            $d28 = substr($data, 124, 8); // 8 郵便番号
            $d29 = substr($data, 132, 6); // 6 RSV
            $d30 = substr($data, 138, 6); // 6 RSV
            $d31 = substr($data, 144, 44); // 44 住所1(1行目)
            $d32 = substr($data, 188, 44); // 44 住所2(2行目)
            $d33 = substr($data, 232, 12); // 12 電話番号
            $d34 = substr($data, 244, 12); // 12 連絡先電話番号
            $d35 = substr($data, 256, 14); // 14 職業
            $d36 = substr($data, 270, 20); // 20 RSV
            $d37 = substr($data, 290, 20); // 20 RSV
            $d38 = substr($data, 310, 44); // 44 メモ(個人コメント)
            $d39 = substr($data, 354, 3); // 3 RSV
            $d40 = substr($data, 357, 1); // 1 RSV
            $d41 = substr($data, 358, 1); // 1 本人/家族
            $d42 = substr($data, 359, 8); // 8 続柄
            $d43 = substr($data, 367, 8); // 8 保険者番号
            $d44 = substr($data, 375, 24); // 24 被保険者証・記号
            $d45 = substr($data, 399, 44); // 44 被保険者証・番号
            $d46 = substr($data, 443, 8); // 8 保険証有効期限
            $d47 = substr($data, 451, 8); // 8 第1公費負担者番号
            $d48 = substr($data, 459, 8); // 8 第1公費受給者番号
            $d49 = substr($data, 467, 8); // 8 第2公費負担者番号
            $d50 = substr($data, 475, 8); // 8 第2公費受給者番号
            $d51 = substr($data, 483, 12); // 12 保険名称
            $d52 = substr($data, 495, 3); // 3 分類A
            $d53 = substr($data, 498, 3); // 3 分類B
            $d54 = substr($data, 501, 3); // 3 分類C
            $d55 = substr($data, 504, 3); // 3 分類D
            $d56 = substr($data, 507, 3); // 3 保険区分
            $d57 = substr($data, 510, 1); // 1 RSV
            $d58 = substr($data, 511, 1); // 1 ETX
            $result = TRUE;
        } elseif ($d1 == $this->c_dle) { // DLE
            $d2 = substr($data, 1, 3);   // 3 識別子
            $tmp = substr($data, 4); // 残り部分(エラーコード+EOT)
            $len = strlen($tmp);
            $d3 = ''; // 5 エラーコード(最大5バイト)
            for ($i = 0; $i < $len; $i++) {
                if ($tmp[$i] === $this->c_eot) { break; } // EOTになったら終了
                $d3 .= $tmp[$i];
            }
            $d4 = $this->c_eot; // 1 EOT
            $code = $d3;
            $result = FALSE;
        } else {
            $result = FALSE;
        }
        $cnt = 58;
        for ($i = 1; $i <= $cnt; $i++) {
            $var = 'd' . $i;
            if (isset($$var)) {
                if ($i != 1 && $i != 26) {
                    $$var = mb_convert_encoding($$var, 'UTF-8', self::fenc);
                }
                $this->_data[$var] = $$var;
            }
        }
        return $result;
    }

    /**
     * Get Request Message
     */
    private function _getRequestMessage($exam_no)
    {
        //return $this->_getRequestAA1($exam_no);
        return $this->_getRequestA11($exam_no);
    }

    private function _getRequestAA1($exam_no)
    {
        $len = strlen($exam_no);
        //$data = self::c_stx;    // 1 STX
        $data = $this->c_stx;    // 1 STX
        //$data = "\x11";    // 1 STX (err test : 010)
        $data .= '00';   // 2 拡張部
        $data .= 'SRD0'; // 4 電文区分
        //$data .= 'SRD2'; // 4 電文区分 (err test : 020)
        $data .= 'E01';  // 3 ブロック区分
        $data .= ' ';    // 1 予備
        $data .= 'AA1';  // 3 データ区分
        //$data .= 'XX1';  // 3 データ区分 (err test : 021)
        $data .= '0';    // 1 サブ区分
        $data .= 'C';    // 1 情報種別
        //$data .= 'X';    // 1 情報種別 (err test : 022)
        $data .= $exam_no . str_repeat(' ', (18 - $len)); // 18 患者コード/患者ID
        $data .= str_repeat(' ', 7); // 7 特定情報
        $data .= str_repeat(' ', 6); // 6 RSV
        //$data .= 'XXXX'; // (err test : 023)
        //$data .= self::c_etx;   // 1 ETX
        $data .= $this->c_etx;   // 1 ETX
        return $data;
    }

    private function _getRequestA11($exam_no)
    {
        $len = strlen($exam_no);
        $data = $this->c_stx;    // 1 STX
        //$data = "\x11";    // 1 STX (err test : 010)
        $data .= '00';   // 2 拡張部
        $data .= 'SRD0'; // 4 電文区分
        //$data .= 'SRD2'; // 4 電文区分 (err test : 020)
        $data .= 'E01';  // 3 ブロック区分
        $data .= ' ';    // 1 予備
        $data .= 'A11';  // 3 データ区分
        //$data .= 'XX1';  // 3 データ区分 (err test : 021)
        $data .= '0';    // 1 サブ区分
        $data .= 'C';    // 1 情報種別
        //$data .= 'X';    // 1 情報種別 (err test : 022)
        $data .= $exam_no . str_repeat(' ', (10 - $len)); // 10 患者コード/患者ID
        $data .= str_repeat(' ', 6); // 6 特定情報
        $data .= str_repeat(' ', 6); // 6 RSV
        //$data .= 'XXXX'; // (err test : 023)
        //$data .= self::c_etx;   // 1 ETX
        $data .= $this->c_etx;   // 1 ETX
        return $data;
    }
    // }}}

}

//$_GET['exam_no'] = '11111111';

// {{{ Main Operation...
$pa = new PatientChecker;
$pa->prepare();
if ($pa->validate()) {
    $pa->execute();
}
$pa->complete();
// }}}

?>
