<?php
/**
 * Framework: Lambda
 *
 * LICENSE: MIT
 *
 * Copyright (c) 2011 rooth. <rooth[dot]lmd[at]gmail[dot]com> http://lmd.root-n.com/
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"), 
 * to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * @author     rooth
 * @copyright  Copyright (c) 2011 rooth. <rooth[dot]lmd[at]gmail[dot]com> http://lmd.root-n.com/
 * @license    http://sourceforge.jp/projects/opensource/wiki/licenses%2FMIT_license MIT License
 * @package    CORE
 */

/**
 * DIRECTORY_SEPARATOR
 */
defined('DS') || define('DS', DIRECTORY_SEPARATOR);

/**
 * error_reporting, display_errors, log_errors, error_log, html_errors
 *  App config location is: Lambda/apps/<APP_DIR>/app.env.php
 *  below config for Lambda.
 */
ini_set('error_reporting', E_ALL | E_STRICT);
ini_set('display_errors',  0);
ini_set('log_errors',  1);
ini_set('error_log',   dirname(__FILE__).DS.'logs'.DS.'Lambda.err');
ini_set('html_errors', 0);

/**
 * フレームワーク： Lambda
 *
 * PHP version 5
 */
class Lambda
{
    const VERSION = '0.0.2';

    protected static $appdir, $mobile_subdir, $mainmodule, $pathinfo;
    protected static $get, $post, $global;

    protected $module, $modular;
    protected $logic, $prep, $view;

    protected $response = array();
    protected $param    = array();
    protected $error    = array();

    const SANITATION   = true;
    const INSANITATION = false;

    const MAX_MODULE = 99;

    const URI_REWRITE_STATIC  = 1;
    const URI_REWRITE_DYNAMIC = 2;
    const URI_REWRITE_HYBRID  = 3;
    const URI_NORMAL          = 9;

    // charset
    protected static $charset = 'UTF-8';
    // env
    private static $env = array();



    ############################################################
    #
    # public static
    #
    ############################################################
    /**
     * バージョン情報を返却する
     * @return str バージョン情報
     */
    public static function getVersion()
    {
        return self::VERSION;
    }

    /**
     * lambda environment.
     * @param array $arr lambda環境設定情報
     */
    public static function setEnv(Array $arr)
    {
        foreach ($arr as $k => $v) {
            if (is_array($v) && isset(self::$env[$k])) {
                self::$env[$k] = array_merge(self::$env[$k], $v);
            } elseif ($k === 'date.timezone') {
                ini_set('date.timezone', $v);
            } else {
                self::$env[$k] = $v;
            }
        }
    }

    /**
     * コンフィグ情報を返却する
     * @param str $key キー
     * @param mixed $fallback 存在しない時の代替
     * @return mixed コンフィグ情報
     */
    public static function getEnv($key = '', $fallback = '')
    {
        if ($key === '') return self::$env;
        return isset(self::$env[$key]) ? self::$env[$key] : $fallback;
    }

    /**
     * グローバル変数を返却する (PHPスーパーグローバルは除く)
     * @return array グローバル変数
     */
    public static function getGlobal()
    {
        if (self::$global === null) self::setGlobal();
        return self::$global;
    }

    /**
     * HTTP HOST を返却する
     * @param boolean $schema https?スキーマを付与するか否か
     * @return str
     */
    public static function httpHost($schema = false)
    {
        $protocol = '';
        if ($schema) $protocol = self::isSSL() ? 'https:/'.'/' : 'http:/'.'/'; // コメント除去対策

        $http_host = self::getServerEnv('HTTP_HOST');
        if ($http_host !== '') {
            return $protocol . $http_host;
        }
        return $protocol . self::getServerEnv('SERVER_NAME');
    }

    /**
     * REQUEST_URI を返却する
     * @return str
     */
    public static function httpRequestUri()
    {
        return self::getServerEnv('REQUEST_URI');
    }

    /**
     * QUERY_STRING を返却する
     * @return str
     */
    public static function httpQueryString()
    {
        return self::getServerEnv('QUERY_STRING');
    }

    /**
     * リクエストされたディレクトリパスを返却する
     *   ※ 末尾の / (フラッシュ) は含めない
     * 
     * @return str
     */
    public static function httpRequestDir()
    {
        static $ret;
        if ($ret !== null) return $ret;

        $req_uri = self::getServerEnv('REQUEST_URI');
        $req_uri = preg_replace('/\?.*$/', '', $req_uri);
        $req_uri = self::stripDummyExt($req_uri);
        $req_uri = preg_replace('/index$/', '', $req_uri);
        if (substr($req_uri, -1) !== '/') {
            $req_uri = preg_replace('|/[^/]+$|', '', $req_uri);
        }

        return $ret = rtrim($req_uri, '/');
    }

    /**
     * SSLか否か
     * @return boolean
     */
    public static function isSSL()
    {
        return (strtolower(self::getServerEnv('HTTPS')) === 'on');
    }

    /**
     * URI(URL)を返却する
     * @param boolean $schema https?スキーマを付与するか否か
     * @return str
     */
    public static function getURI($schema = true)
    {
        return self::httpHost($schema) . self::httpRequestUri();
    }
    public static function getURL($schema = true)
    {
        return self::getURI($schema);
    }

    /**
     * 出力する (正規表現にて、文字列置換を行う機能も併せ持つ)
     * @param  str $output 出力内容
     */
    public static function output($output)
    {
        if ($filter = self::getEnv('display_filter')) {
            foreach ($filter as $regex => $replacement) $output = preg_replace($regex, $replacement, $output);
        }
        self::sendHttpHeader();
        echo $output;
    }



    ############################################################
    #
    # protected static
    #
    ############################################################
    /**
     * モバイルからのリクエストか否か
     */
    protected static function isMobile()
    {
        return (bool)self::getEnv('is_mobile');
    }

    /**
     * アプリケーションディレクトリをセットする
     * @param str $dir アプリケーションディレクトリ
     */
    protected static function setAppDir($dir)
    {
        if (self::$appdir !== null) return;
        self::$appdir = rtrim($dir, DS);
    }

    /**
     * アプリケーションディレクトリを返却する
     * @return str アプリケーションディレクトリ
     */
    protected static function getAppDir()
    {
        return self::$appdir;
    }

    /**
     * モバイルサブディレクトリをセットする
     * @param str $dir モバイルサブディレクトリ
     */
    protected static function setMobileSubDir($dir)
    {
        if (self::$mobile_subdir !== null) return;
        self::$mobile_subdir = trim($dir, DS);
    }

    /**
     * モバイルサブディレクトリを返却する
     * @return str モバイルサブディレクトリ
     */
    protected static function getMobileSubDir()
    {
        if (self::$mobile_subdir === null) {
            self::setMobileSubDir(self::getEnv('mobile_subdir'));
        }
        return self::$mobile_subdir;
    }

    /**
     * メインモジュール名をセットする
     * @param str $path URL形式の文字列
     */
    protected static function setMainModule($path)
    {
        if (self::$mainmodule !== null) return;
        self::$mainmodule = self::normalizeModule($path);
    }

    /**
     * グローバル変数をセットする (PHPスーパーグローバルは除く)
     */
    protected static function setGlobal()
    {
        if (self::$global !== null) return;
        $ignoral = array('_GET', '_POST', '_REQUEST', '_COOKIE', '_SESSION', '_SERVER', '_FILES', '_ENV');
        self::$global = array();
        foreach ($GLOBALS as $k => $v) {
            if (strpos($k, '_') === 0 && ! in_array($k, $ignoral)) self::$global[$k] = $v;
        }
    }

    /**
     * スーパーグローバル: PATH_INFO をセットする
     */
    protected static function setPathInfo()
    {
        if (self::$pathinfo !== null) return;

        $pi = getenv('PATH_INFO') ? getenv('PATH_INFO') : getenv('ORIG_PATH_INFO');
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $pi = preg_replace('|^.*?/index\.php|', '', $pi);
        }

        self::$pathinfo = $pi ? $pi : '';
    }

    /**
     * スーパーグローバル: PATH_INFO を返却する
     * @return str $_SERVER['ORIG_PATH_INFO'] or $_SERVER['PATH_INFO'] or ''
     */
    protected static function getPathInfo()
    {
        if (self::$pathinfo === null) self::setPathInfo();
        return self::$pathinfo;
    }

    /**
     * リクエスト($_GET, $_POST)をセットする
     */
    protected static function setRequest()
    {
        if (self::$get !== null) return;

        self::$get  = array();
        self::$post = array();

        switch (self::getEnv('uri_notation')) {
            case self::URI_REWRITE_STATIC:
            case self::URI_REWRITE_HYBRID:
                // fetch query string.
                $qs = self::stripDummyExt(self::getQueryString());

                $active_key = '';
                foreach (explode(self::getEnv('uri_delim_amp'), $qs) as $val) {
                    if ($val === '') continue;
                    if (strpos($val, self::getEnv('uri_delim_eq')) !== false) {
                        list($k, $v) = explode(self::getEnv('uri_delim_eq'), $val, 2);
                        $active_key = $k;
                    } else {
                        if ($uri_sticky_params = self::getEnv('uri_sticky_params')) {
                            if (in_array($active_key, $uri_sticky_params)) {
                                $k = $active_key;
                                $v = self::$get[$active_key] . '/' . $val;
                            }
                        } else {
                            list($k, $v) = array($val, '');
                        }
                    }

                    self::$get[$k] = $v;
                }
        }

        switch (self::getEnv('uri_notation')) {
            case self::URI_REWRITE_STATIC:
                break;

            case self::URI_REWRITE_HYBRID:
                //if (isset($_GET) && $_GET) self::$get += $_GET;
                if (isset($_GET) && $_GET) self::$get = array_merge(self::$get, $_GET);
                break;

            case self::URI_REWRITE_DYNAMIC:
            case self::URI_NORMAL:
            default:
                if (isset($_GET) && $_GET) self::$get = $_GET;
        }

        if (isset($_POST) && $_POST) self::$post = $_POST;
    }

    /**
     * リクエストパス を返却する
     * @return str リクエストパス
     */
    protected static function getRequestPath()
    {
        list($path, ) = self::splitPathInfo();
        return $path;
    }

    /**
     * クエリーストリング を返却する
     * @return str クエリーストリング
     */
    protected static function getQueryString()
    {
        list( , $querystring) = self::splitPathInfo();
        return $querystring;
    }

    /**
     * PathInfo を path と querystring に分割する
     * @return array array(path, querystring)
     */
    protected static function splitPathInfo()
    {
        $pathinfo = self::getPathInfo();
        if (strpos($pathinfo, self::getEnv('uri_delim_ques')) === false) {
            return array($pathinfo, '');
        } else {
            list($path, $qs) = explode(self::getEnv('uri_delim_ques'), $pathinfo, 2);
            return array($path, $qs);
        }
    }

    /**
     * ダミー拡張子を除去する
     * @param  str $str 対象文字列
     * @return str ダミー拡張子を除去した文字列
     */
    protected static function stripDummyExt($str)
    {
        if (self::getEnv('uri_dummy_ext') !== '') {
            $ext = ltrim(self::getEnv('uri_dummy_ext'), '.');
            $regex = '/\.'.$ext.'$/';
            return preg_replace($regex, '', $str);
        }
        return $str;
    }

    /**
     * モジュール名の妥当性検証
     * @param  str $module モジュール名
     */
    protected static function inspectModule($module)
    {
        if (strpos($module, '..') !== false || ! preg_match('#^[0-9a-zA-Z_.]*$#', $module)) {
            $msg = 'Fatal: invalid module `'.$module."': you can use ^[0-9a-zA-Z_]*$ as module.";
            self::terminate($msg, 400);
        }
    }

    /**
     * モジュール名の正規化
     * @param str $path URL形式の文字列
     * @return str モジュール名
     */
    protected static function normalizeModule($path)
    {
        $path = self::stripDummyExt($path);
        if (strpos($path, '.') !== false) {
            self::terminate("Fatal: invalid url: can't use dot character into module."+$path, 400);
        }
        $path = str_replace('/', '.', ltrim($path, '/'));

        // fetch module name.
        list($mod, ) = explode(self::getEnv('uri_delim_ques'), $path, 2);
        $mod = ($mod === null) ? '' : $mod;

        self::inspectModule($mod);

        if (strpos($mod, '.') !== false) {
            if (substr($mod, -1) === '.' && substr_count($mod, '.') === 1) {
                // ex) example.com/a/ => a.index
                $module = str_replace('.', '', $mod).self::getEnv('def_module_member');
            } elseif (substr($mod, -1) === '.' && substr_count($mod, '.') > 1) {
                // ex) example.com/a/b/ => a.b
                $module = preg_replace('#\.+$#', '', $mod);
            } else {
                // ex) example.com/a/b/c => a.b.c
                $module = $mod;
            }
        } elseif ($mod !== '') {
            // ex) example.com/a => a.index
            $module = $mod.self::getEnv('def_module_member');
        } else {
            // ex) example.com/ => index.index
            $module = self::getEnv('def_module_group').self::getEnv('def_module_member');
        }

        return $module;
    }

    /**
     * サフィックスを返却する
     * @param  str $modular モジュラー
     * @return str サフィックス ex) .view.php
     */
    protected static function getSuffix($modular)
    {
        $arr  = explode('.', $modular);
        $ext  = array_pop($arr); // pop: php
        $type = array_pop($arr);
        return ".{$type}.{$ext}";
    }

    /**
     * サーバー環境変数を返却する
     * @param  str $varname 変数名
     * @return str 環境変数
     */
    protected static function getServerEnv($varname)
    {
        return isset($_SERVER[$varname]) ? $_SERVER[$varname] : getenv($varname);
    }

    /**
     * 強制終了させる
     * @param str $msg メッセージ
     * @param int $sig シグナル
     */
    protected static function terminate($msg, $sig = -1)
    {
        if (self::isDisplayErrors()) echo $msg."\n\n";

        switch ($sig) {
        case 400: // Bad Request 
            $lmd = new self;
            $lmd->invokeModule(self::getEnv('def_400mod_group').self::getEnv('def_module_member'));
            exit;

        case 404: // Not Found
            $lmd = new self;
            $lmd->invokeModule(self::getEnv('def_404mod_group').self::getEnv('def_module_member'));
            exit;

        case -1:
        default:
            exit;
        }
    }

    /**
     * recursive htmlspecialchars
     */
    protected static function rh($mixed)
    {
        if (is_scalar($mixed) && ! is_bool($mixed)) {
            return htmlspecialchars($mixed, ENT_QUOTES, self::$charset);
        } elseif (is_array($mixed)) {
            return array_map(array('self', 'rh'), $mixed);
        } else {
            return $mixed;
        }
    }

    /**
     * recursive ntrim
     */
    protected static function ntrim($mixed)
    {
        if (is_scalar($mixed) && ! is_bool($mixed)) {

            // 制御文字(TAB, LF, CR, SPACE以外)をすべて除去
            // (x09: TAB / x0a: LF / x0d: CR / x20: SPACE)
            $mixed = preg_replace('/[\x00-\x08\x0b\x0c\x0e-\x1f\x7f]/u', '', $mixed);

            // 前後のホワイトスペース、全角スペースを除去
            $mixed = preg_replace('/^[\s　]+|[\s　]+$/u', '', $mixed);

            return $mixed;

        } elseif (is_array($mixed)) {
            return array_map(array('self', 'ntrim'), $mixed);
        } else {
            return $mixed;
        }
    }

    /**
     * HTTP HEADER を送出する
     */
    protected static function sendHttpHeader()
    {
        foreach (self::getEnv('http_header', array()) as $k => $v) header($k.': '.$v);
    }

    /**
     * エラー表示モードの判定
     */
    protected static function isDisplayErrors()
    {
        return (bool)ini_get('display_errors');
    }



    ############################################################
    #
    # public
    #
    ############################################################
    /**
     * constructor
     * @param str $appdir アプリケーションディレクトリ
     */
    public function __construct($appdir = null)
    {
        // constructor is executed only once. 
        if (self::getAppDir() !== null) return;

        // set app dir.
        self::setAppDir($appdir);

        // default configuration.
        self::setEnv(array(
            // uri_*
            'uri_notation'   => self::URI_NORMAL,
            'uri_dummy_ext'  => '',
            'uri_delim_ques' => '-', // ?
            'uri_delim_amp'  => '/', // &
            'uri_delim_eq'   => '-', // =

            // def_*
            'def_module_group'  => 'index',  // default module group
            'def_400mod_group'  => '400',    // default 400 module group (400 Bad Request)
            'def_404mod_group'  => '404',    // default 404 module group (404 Not Found)
            'def_module_member' => '.index', // default module member

            // suff_*
            'suff_logic' => '.logic.php',
            'suff_prep'  => '.prep.php',
            'suff_view'  => '.view.php',
            'suff_func'  => '.func.php',

            // mobile
            'is_mobile'     => false,
            'mobile_subdir' => '',

            // session_start() ?
            'session_start' => false,

            'max_module' => self::MAX_MODULE,
            'validator_dir' => dirname(__FILE__).DS.'validators',
        ));

        // require `app.env.php' if necessary. and overwriting.
        $env = self::getAppDir().DS.'app.env.php';
        is_readable($env) && require $env;

        // set_error_handler, register_shutdown_function
        if (defined('LAMBDA_ENABLE_CUSTOM_ERROR') && LAMBDA_ENABLE_CUSTOM_ERROR) {
            set_error_handler('lambda_custom_error_handler', 
                defined('LAMBDA_ERROR_REPORTING') ? LAMBDA_ERROR_REPORTING : E_ALL
            );
            if (version_compare(PHP_VERSION, '5.2.0', '>=')) {
                register_shutdown_function('lambda_custom_shutdown_handler');
            }
        }

        // set static vars: mainmodule, request.
        self::setMainModule(self::getRequestPath());
        self::setRequest();

        // set module.
        $this->setModule($this->getMainModule());

        // session start.
        self::getEnv('session_start') && session_start();
    }

    /**
     * メインモジュール名を返却する
     * @return str メインモジュール名
     */
    public function getMainModule()
    {
        return self::$mainmodule;
    }

    /**
     * メインモジュールの判定
     * @param str $mod モジュール名
     */
    public function isMainModule($mod)
    {
        return (bool)((string)$this->getMainModule() === (string)$mod);
    }

    /**
     * メインモジュールのグループ名を返却する
     * @return str メインモジュールのグループ名
     */
    public function getMainModuleGroup()
    {
        list($group, ) = explode('.', self::getMainModule(), 2);
        return $group;
    }

    /**
     * メインモジュールグループの判定
     * @param str $grp グループ名
     * @return boolean
     */
    public function isMainModuleGroup($grp)
    {
        return (bool)((string)$this->getMainModuleGroup() === (string)$grp);
    }

    /**
     * モジュール名を返却する
     * @return str モジュール名
     */
    public function getModule()
    {
        return $this->module;
    }

    /**
     * モジュラーを返却する
     * @param  str $module モジュール名
     * @param  str $suffix 接尾辞
     * @return str モジュラー
     */
    public function getModular($module, $suffix)
    {
        list($group, $member) = explode('.', $module, 2);

        $modulars = array();
        if (self::isMobile()) {
            $modulars[] = $group.DS.self::getMobileSubDir().DS.$member.$suffix;
            $modulars[] = $group.DS.self::getMobileSubDir().DS.$member.self::getEnv('def_module_member').$suffix;
        }
        $modulars[] = $group.DS.$member.$suffix;
        $modulars[] = $group.DS.$member.self::getEnv('def_module_member').$suffix;

        foreach ($modulars as $modular) {
            if (is_readable(self::getAppDir().DS.$modular)) return self::getAppDir().DS.$modular;
        }
        return '';
    }

    /**
     * ロジック名をセットする
     * @param str $logic ロジック名
     */
    public function setLogic($logic)
    {
        $this->logic = $logic;
    }

    /**
     * ロジック名を返却する
     * @return str ロジック名
     */
    public function getLogic()
    {
        return $this->logic;
    }

    /**
     * プレップ名をセットする
     * @param str $prep プレップ名
     */
    public function setPrep($prep)
    {
        $this->prep = $prep;
    }

    /**
     * プレップ名を返却する
     * @return str プレップ名
     */
    public function getPrep()
    {
        return ($this->prep !== null) ? $this->prep : $this->getLogic();
    }

    /**
     * ビュー名をセットする
     * @param str $view ビュー名
     */
    public function setView($view)
    {
        $this->view = $view;
    }

    /**
     * ビュー名を返却する
     * @return str ビュー名
     */
    public function getView()
    {
        return ($this->view !== null) ? $this->view : $this->getPrep();
    }

    /**
     * レスポンスをセットする
     * @param array $arr レスポンスデータ
     * @param bool  $sanitize サニタイズするか否か
     */
    public function setResponse(Array $arr, $sanitize = self::SANITATION)
    {
        foreach ($arr as $k => $v) {
            if ($k === 'lmd') {
                self::terminate("Fatal: response key name `lmd' is reserved.");
            }
            $this->response[$k] = ($sanitize) ? self::rh($v) : $v;
        }
    }

    /**
     * レスポンスを返却する
     * @param str   $key レスポンスのキー名
     * @param mixed $fallback 存在しない時の代替
     * @return mixed レスポンス値
     */
    public function getResponse($key = '', $fallback = '')
    {
        if ($key === '') return $this->response;
        return isset($this->response[$key]) ? $this->response[$key] : $fallback;
    }

    /**
     * エラーをセットする
     * @param mixed $mixed エラーデータ
     * @param bool  $sanitize サニタイズするか否か
     */
    public function setError($mixed, $sanitize = self::SANITATION)
    {
        $mixed = ($sanitize) ? self::rh($mixed) : $mixed;

        if (is_array($mixed)) foreach ($mixed as $k => $v) {
            preg_match('/^[0-9]+$/', $k) ? $this->error[] = $v : $this->error[$k] = $v;
        } else {
            $this->error[] = $mixed;
        }
    }

    /**
     * エラーを返却する
     * @param str   $key エラーのキー名
     * @param mixed $fallback 存在しない時の代替
     * @return mixed エラー値
     */
    public function getError($key = '', $fallback = '')
    {
        if ($key === '') return $this->error;
        return isset($this->error[$key]) ? $this->error[$key] : $fallback;
    }

    /**
     * パラメータ値を返却する
     * @param str   $key パラメータのキー名
     * @param mixed $fallback 存在しない時の代替
     * @param str   $regex 正規表現 (マッチした場合のみ返却したい時に利用。スカラー変数のみ対応)
     * @return mixed パラメータ値
     */
    public function param($key = '', $fallback = '', $regex = '')
    {
        return $this->getRequest('param', $key, $fallback, $regex);
    }

    /**
     * ゲットリクエスト($_GET)を返却する
     *
     *  spec: 1) 制御文字(TAB, LF, CR, SPACE以外)はすべて除去される
     *        2) 前後のホワイトスペース、全角スペースは除去される
     *
     * @param str   $key ゲットリクエストのキー名
     * @param mixed $fallback 存在しない時の代替
     * @param str   $regex 正規表現 (マッチした場合のみ返却したい時に利用。スカラー変数のみ対応)
     * @return mixed ゲットリクエスト
     */
    public function httpGet($key = '', $fallback = '', $regex = '')
    {
        return self::ntrim($this->rawHttpGet($key, $fallback, $regex));
    }
    /**
     * 未加工のゲットリクエスト($_GET)を返却する
     */
    public function rawHttpGet($key = '', $fallback = '', $regex = '')
    {
        return $this->getRequest('get', $key, $fallback, $regex);
    }

    /**
     * ポストリクエスト($_POST)を返却する
     *
     *  spec: 1) 制御文字(TAB, LF, CR, SPACE以外)はすべて除去される
     *        2) 前後のホワイトスペース、全角スペースは除去される
     *
     * @param str   $key ポストリクエストのキー名
     * @param mixed $fallback 存在しない時の代替
     * @param str   $regex 正規表現 (マッチした場合のみ返却したい時に利用。スカラー変数のみ対応)
     * @return mixed ポストリクエスト
     */
    public function httpPost($key = '', $fallback = '', $regex = '')
    {
        return self::ntrim($this->rawHttpPost($key, $fallback, $regex));
    }
    /**
     * 未加工のポストリクエスト($_POST)を返却する
     */
    public function rawHttpPost($key = '', $fallback = '', $regex = '')
    {
        return $this->getRequest('post', $key, $fallback, $regex);
    }

    /**
     * ゲットリクエスト($_GET)の存在確認
     * @param str $key ゲットリクエストのキー名
     * @return boolean
     */
    public function hasHttpGet($key)
    {
        return ($this->getRequest('get', $key, null) !== null);
    }

    /**
     * ポストリクエスト($_POST)の存在確認
     * @param str $key ポストリクエストのキー名
     * @return boolean
     */
    public function hasHttpPost($key)
    {
        return ($this->getRequest('post', $key, null) !== null);
    }

    /**
     * クッキーの取得/保存
     * @param str $key クッキーのキー名
     * @param str $val 保存する値
     * @param array $p array(expire, path, domain, secure);
     */
    public function httpCookie($key, $val = null, Array $p = array())
    {
        if ($val === null) {
            // cookie: get
            return isset($_COOKIE[$key]) ? $_COOKIE[$key] : '';
        } else {
            // cookie: set
            $expire = isset($p['expire']) ? $p['expire'] : 0; // Epoc / def:0 ブラウザを閉じたとき無効
            $path   = isset($p['path']) ? $p['path'] : ''; // str / def:クッキーがセットされたときのカレントディレクトリ
            $domain = isset($p['domain']) ? $p['domain'] : '';
            $secure = isset($p['secure']) ? $p['secure'] : false; // bool / def:false
            //$httponly = isset($p['httponly']) ? $p['httponly'] : false; // bool / def:false
            return setcookie($key, $val, $expire, $path, $domain, $secure);
        }
    }

    /**
     * クッキーの存在確認
     * @param str $key クッキーのキー名
     * @return boolean
     */
    public function hasHttpCookie($key)
    {
        return isset($_COOKIE[$key]);
    }

    /**
     * モジュールを実行する
     * @param  str   $module モジュール名
     * @param  array $params パラメーター
     * @param  bool  $logic Logicモジュラーを呼び出すか否か
     * @param  bool  $function Functionモジュラーを呼び出すか否か
     * @return str   出力用文字列
     */
    public function invokeModule($module, Array $params = array(), $logic = true, $function = false)
    {
        self::inspectModule($module);

        static $module_counter = 0;
        $module_counter++;
        if (0 < self::getEnv('max_module') && self::getEnv('max_module') <= $module_counter) {
            self::terminate('Fatal: Maximum execution module of '.$module_counter.' exceeded.');
        }
        if ($this->getMainModule() !== $module && $this->getModule() === $module) {
            self::terminate('Fatal: Can not redeclare module in same module: `'.$module."'");
        }

        $lmd = new self;
        $lmd->setModule($module);
        $lmd->setLogic($module);
        $lmd->setParam($params);

        // invoke function.
        if ($function) {
            if ($modular = $lmd->getModular($lmd->getLogic(), self::getEnv('suff_func'))) {
                return $lmd->invoke($lmd, $modular);
            } else {
                self::terminate('Fatal: Not found function: `'.$module.self::getEnv('suff_func')."'");
            }
        }

        // invoke logic modular.
        if ($logic) {
            if ($modular = $lmd->getModular($lmd->getLogic(), self::getEnv('suff_logic'))) $lmd->invoke($lmd, $modular);
            if ($modular && $lmd->getLogic() !== $lmd->getModule()) {
                // logic -> logic
                $lmd->invokeModule($lmd->getLogic(), $params);
                return;
            }
        }

        // invoke prep modular.
        if ($modular = $lmd->getModular($lmd->getPrep(), self::getEnv('suff_prep'))) $lmd->invoke($lmd, $modular);
        /*if ($modular && $lmd->getLogic() !== $lmd->getModule()) {
            // prep -> logic
            $lmd->invokeModule($lmd->getPrep(), $params);
            return;
        }
        if ($modular && $lmd->getPrep() !== $lmd->getModule()) {
            // prep -> prep
            $lmd->invokeModule($lmd->getPrep(), $params, false);
            return;
        }*/

        // invoke view modular.
        ob_start();
        ob_implicit_flush(false);
        if ($modular = $lmd->getModular($lmd->getView(), self::getEnv('suff_view'))) $lmd->invoke($lmd, $modular);
        else {
            if ($this->getMainModule() !== $lmd->getModule()) {
                // Module Not Found.
                if (self::isDisplayErrors()) echo '(Module Not Found: '.$lmd->getModule().')';
            } else {
                // 404 Not Found.
                $lmd->invokeModule(self::getEnv('def_404mod_group').self::getEnv('def_module_member'), array(
                    'module' => $lmd->getView()
                ));
            }
        }
        $buff = ob_get_clean();

        // return or echo.
        if ($this->getMainModule() === $module) {
            return $buff;
        } else {
            echo $buff;
        }
    }

    /**
     * ファンクションを実行する
     * @param  str   $module モジュール名
     * @param  array $params パラメーター
     * @return mixed ファンクションの実行結果
     */
    public function invokeFunction($module, Array $params = array())
    {
        return $this->invokeModule($module, $params, false, true);
    }

    /**
     * バリデーターを実行する
     * @param  str   $script バリデータースクリプト
     * @param  array $arr 検証する値を格納した配列(ハッシュ)
     * @return bool
     */
    public function invokeValidator($script, Array $arr)
    {
        $___validator___ = self::getEnv('validator_dir').DS."{$script}.php";
        if ( ! is_readable($___validator___)) {
            self::terminate("Fatal: Not found validator script (or Permission denied): `{$___validator___}'");
        }
        extract($arr);
        extract(array('lmd' => $this), EXTR_SKIP);
        if ( ! $lmd instanceof Lambda) {
            self::terminate("Fatal: validator variable `\$lmd' is reserved.");
        }
        require $___validator___;
    }

    /**
     * バリデーターエラーの存在確認
     * @return bool
     */
    public function hasValidatorError()
    {
        return (bool)$this->getError();
    }

    /**
     * モジュラーを実行する
     * @param  obj $lmd ラムダオブジェクト
     * @param  str $modular モジュラー
     */
    public function invoke(Lambda $lmd, $modular)
    {
        $___modular___ = $modular;

        if (self::getSuffix($___modular___) === self::getEnv('suff_view')) {
            extract($lmd->getResponse());
        }
        extract(self::getGlobal());
        return require $___modular___;
    }

    /**
     * リダイレクトさせる
     * @param str $url リダイレクト先URL or PATH(/からの絶対PATH)
     * @param int $response_code HTTPレスポンスコード (300台)
     * @see http://ja.wikipedia.org/wiki/HTTP%E3%82%B9%E3%83%86%E3%83%BC%E3%82%BF%E3%82%B9%E3%82%B3%E3%83%BC%E3%83%89
     */
    public function redirect($url, $response_code = 302)
    {
        /**
         * 301 Moved Permanently
         * 302 Found
         * 303 See Other
         * 307 Temporary Redirect
         */
        if ( ! preg_match('|^https?:\/\/|', $url)) {
            // rfc準拠：リダイレクトは絶対URL
            $url = $this->httpHost(true).'/'.ltrim($url, '/');
        }
        header('Location: '.$url, true, $response_code);
        exit;
    }



    ############################################################
    #
    # protected
    #
    ############################################################
    /**
     * モジュール名をセットする
     * @param str $module モジュール名
     */
    protected function setModule($module)
    {
        $this->module = $module;
    }

    /**
     * リクエスト($_GET, $_POST, $param)を返却する
     * @param str   $var `get' or `post' or `param'
     * @param str   $key リクエストのキー名
     * @param mixed $fallback 存在しない時の代替
     * @param str   $regex 正規表現 (マッチした場合のみ返却したい時に利用。スカラー変数のみ対応)
     * @return mixed リクエスト
     */
    protected function getRequest($var, $key, $fallback, $regex = '')
    {
        switch ($var) {
            case 'get':
                $var =& self::$get;
                break;

            case 'post':
                $var =& self::$post;
                break;

            case 'param':
                $var =& $this->param;
                break;

            default:
                self::terminate('Fatal: invalid parameter $var.');
        }

        if ($key === '') return $var;

        if ( ! isset($var[$key])) return $fallback;

        if ($regex !== '' && is_scalar($var[$key])) {
            if ( ! preg_match($regex, $var[$key])) return $fallback;
        }

        return $var[$key];
    }

    /**
     * パラメータ値をセットする
     * @param array $param パラメータ
     */
    protected function setParam(Array $param)
    {
        $this->param = $param;
    }



    /*
      error messages.

        (Module Not Found: hogehoge)
        Fatal: invalid url: can't use dot character into module.
        Fatal: invalid module `hoge*': you can use ^[0-9a-zA-Z_]*$ as module.
        Fatal: Maximum execution module of 99 exceeded.
        Fatal: Can not redeclare module in same module: `hogehoge'
        Fatal: Not found function: `hogehoge.func.php'
        Fatal: Not found validator script (or Permission denied): `/path/to/validator.php'
        Fatal: response key name `lmd' is reserved.
        Fatal: validator variable `$lmd' is reserved.

        Fatal: Not defined LAMBDA_APPDIR.
        Fatal: Not found (or Permission denied): /path/to/Lambda/apps/app
        Fatal: Not found (or Permission denied): /path/to/Lambda.init.php
    */



} // --- End of class



/**
 * main
 */
if ( ! defined('LAMBDA_APPDIR'))   exit('Fatal: Not defined LAMBDA_APPDIR.');
if ( ! is_readable(LAMBDA_APPDIR)) exit('Fatal: Not found (or Permission denied): '.LAMBDA_APPDIR);

// proc1: require `Lambda.init.php'
if (is_readable(LAMBDA_BASEDIR.DS.'Lambda.init.php')) {
    require LAMBDA_BASEDIR.DS.'Lambda.init.php';
} else  {
    exit('Fatal: Not found (or Permission denied): '.LAMBDA_BASEDIR.DS.'Lambda.init.php');
}

// proc2: require `apps.conf.php' if necessary.
if (is_readable(LAMBDA_BASEDIR.DS.'apps'.DS.'apps.conf.php')) {
    require LAMBDA_BASEDIR.DS.'apps'.DS.'apps.conf.php';
}

// proc3: require `app.conf.php' if necessary.
if (is_readable(LAMBDA_APPDIR.DS.'app.conf.php')) {
    require LAMBDA_APPDIR.DS.'app.conf.php';
}

// proc4: construct Lambda.
//     4.1 app.env.php
//     4.2 ::setMainModule()
//     4.3 ::setRequest()
//     4.4 ->setModule()
//     4.5 session_start() if necessary.
$lmd = new Lambda(LAMBDA_APPDIR);

// proc5: require `apps.init.php' if necessary.
if (is_readable(LAMBDA_BASEDIR.DS.'apps'.DS.'apps.init.php')) {
    require LAMBDA_BASEDIR.DS.'apps'.DS.'apps.init.php';
}

// proc6: require `app.init.php' if necessary.
if (is_readable(LAMBDA_APPDIR.DS.'app.init.php')) {
    require LAMBDA_APPDIR.DS.'app.init.php';
}

// proc7: invoke main module and output.
Lambda::output(
    $lmd->invokeModule($lmd->getMainModule())
);

// proc8: require `app.fini.php' if necessary.
if (is_readable(LAMBDA_APPDIR.DS.'app.fini.php')) {
    require LAMBDA_APPDIR.DS.'app.fini.php';
}

// proc9: require `apps.fini.php' if necessary.
if (is_readable(LAMBDA_BASEDIR.DS.'apps.fini.php')) {
    require LAMBDA_BASEDIR.DS.'apps.fini.php';
}
