<?php

namespace doctor_tab_outpatient_carte_logic;

use \Lambda;
use \Factory;
use \Resource;
use \Severity;

// outpatient_carte_id
$oc_id = $lmd->httpGet('oc_id');

// 外来カルテ情報を取得
{
    $dao    = Factory::load('DB/DAO_outpatient_carte')->getInstance();
    $cartes = $dao->fetchAll_BY_answer_code($_answer_code);

    if ($oc_id) {
        $carte = $dao->fetch_BY_id($oc_id);
    } else {
        $carte = $cartes ? $cartes[0] : array();
    }
}

// 患者様の回答情報
$answer = getAnswer();

// 症状情報
$diseases = getDiseases($answer);

// 救急内因系診断名
$ex_diagnosis = getInternalDiagnosis();
// 救急外傷系診断名
$em_diagnosis = getEmergencyInjuryDiagnosis();
$em_category  = getEmergencyInjuryDiagnosisCategory();

// set response.
$lmd->setResponse(array(
    'oc_id' => $oc_id,
    'historys' => getHistorys($cartes), // 履歴プルダウン
    'ex_diagnosis' => $ex_diagnosis,
    'serious_level' => $answer['serious_level'], // 外傷重症度
    'em_diagnosis' => $em_diagnosis, // 救急外傷系診断名(Ajaxは使わないで取得用)
    'em_category' => $em_category,   // 救急外傷系診断名カテゴリ
));
if (isset($carte['data'])) {
    // 過去に入力した外来カルテ情報あり
    $def_json = array_merge((array)json_decode($carte['data']), getVitalSign($answer));
    $lmd->setResponse(array(
        'is_history' => TRUE,
        'json' => json_encode($def_json),
        'visible_items' => array(), // 表示するカルテ大項目
    ), Lambda::INSANITATION);
} else {
    // 外来カルテ情報なし
    $lmd->setResponse(array(
        'is_history' => FALSE,
        'json' => json_encode(array_merge(getDefault($answer, $diseases), getVitalSign($answer))),
        'visible_items' => json_encode(getVisibleItems($diseases, $answer)), // 表示するカルテ大項目
    ), Lambda::INSANITATION);
}



/**
 * 救急内因系診断名の取得
 */
function getInternalDiagnosis()
{
    $dao = Factory::load('DB/DAO_external_injury_diagnosis')->getInstance();
    $diagnosis = $dao->fetchAllExternalInjuryDiagnosis();
    $result = array();
    foreach ($diagnosis as $val) {
        $result[$val['external_injury_diagnosis_id']] = $val['diagnosis_name'];
    }
    return $result;
}

/**
 * 救急外傷系診断名カテゴリの取得
 */
function getEmergencyInjuryDiagnosisCategory()
{
    $dao = Factory::load('DB/DAO_emergency_injury_diagnosis_category')->getInstance();
    $diagnosis = $dao->fetchAllEmergencyInjuryDiagnosisCategory();
    $result = array();
    foreach ($diagnosis as $val) {
        $result[$val['emergency_injury_diagnosis_category_id']] = $val['category_name'];
    }
    return $result;
}

/**
 * 救急外傷系診断名の取得
 */
function getEmergencyInjuryDiagnosis()
{
    $dao = Factory::load('DB/DAO_emergency_injury_diagnosis')->getInstance();
    $diagnosis = $dao->fetchAllEmergencyInjuryDiagnosis();
    $result = array();
    foreach ($diagnosis as $val) {
        $result[] = array(
                    'id'    => $val['emergency_injury_diagnosis_id'],
                    'name'  => $val['diagnosis_name'],
                    'catid' => $val['emergency_injury_diagnosis_category_id'],
                    );
    }
    return $result;
}

/**
 * 患者様の回答情報を取得
 */
function getAnswer()
{
    global $_answer_code;

    $dao = Factory::load('DB/DAO_answers')->getInstance();
    $answer = $dao->fetchNewest_LEFT_JOIN_severitys_BY_answer_code($_answer_code);

    return $answer;
}

/**
 * 症状を取得
 */
function getDiseases(Array $answer)
{
    $shojo_category_id = $answer['shojo_category_id'];
    $disease_ids = $answer['step3_answers'] ? explode(',', $answer['step3_answers']) : array();

    $diseases = array();
    $dao = getCarteDisRelationTableDao();
    foreach ($disease_ids as $disease_id) {
        $diseases[] = $dao->fetch_BY_CategoryID_AND_DiseaseID($shojo_category_id, $disease_id);
    }

    return $diseases;
}

/**
 * ログイン者のロールと対象患者(小児科or内科)に対する CarteDisRelationTableN の DAO を返却する
 */
function getCarteDisRelationTableDao()
{
    if (_isChild()) {
        // 小児科(16歳未満)
        if (_isRoleDoctor()) {
            // 小児の医師・小児科医
            $dao = Factory::load('DB/DAO_CarteDisRelationTable01')->getInstance();
        } else {
            // 小児の医師・小児科医以外
            $dao = Factory::load('DB/DAO_CarteDisRelationTable02')->getInstance();
        }
    } else {
        // 内科(16歳以上)
        if (_isRoleDoctor()) {
            // 内科の医師・小児科医
            $dao = Factory::load('DB/DAO_INT_CarteDisRelationTable01')->getInstance();
        } else {
            // 内科の医師・小児科医以外
            $dao = Factory::load('DB/DAO_INT_CarteDisRelationTable02')->getInstance();
        }
    }

    return $dao;
}

/**
 * 症状から表示するカルテ大項目を取得
 */
function getVisibleItems(Array $diseases, Array $answer)
{
    $buf = '';
    foreach ($diseases as $v) {
        $buf .= $v['DefaultCheckFlag'];
    }

    $visible_items = array();
    foreach (explode('#', $buf) as $v) {
        $v = trim($v);
        if ( ! $v) continue;
        if ( ! in_array($v, $visible_items)) $visible_items[] = $v;
    }

    /*
    ①追加する大項目の処理

    現在37.5℃以上で0日以上546日未満の場合、Head追加
    #27000#

    現在37.5℃以上で181日以上の場合、Kernig sign追加
    #10000#

    現在37.5℃以上で91日以上の場合、Neck stiffness追加
    #9000#
    */
    if ((float)$answer['nowtemp'] >= 37.5) {
        if (0 <= $answer['life_days'] && $answer['life_days'] < 546) {
            if ( ! in_array(27000, $visible_items)) $visible_items[] = 27000;
        }
        if (181 <= $answer['life_days']) {
            if ( ! in_array(10000, $visible_items)) $visible_items[] = 10000;
        }
        if (91 <= $answer['life_days']) {
            if ( ! in_array(9000, $visible_items)) $visible_items[] = 9000;
        }
    }

    return $visible_items;
}

/**
 * 【 Onset and Course 】の内容を返却する
 */
function getOnsetAndCourse(Array $answer, Array $diseases)
{
    $ret = '';
    $when_arr = Resource::get('when');
    $buf = $answer['step31_when'] ? explode(',', $answer['step31_when']) : array();

    // Sort and Convert to Array...
    $sort = array();
    $dise = array();
    foreach ($buf as $v) {
        list($disease_id, $when_key) = explode(':', $v);
        $key = $when_key . '_' . $disease_id;
        $sort[] = $key;
        $dise[$key] = array('disease_id' => $disease_id, 'when_key' => $when_key);
    }
    rsort($sort);
    $done = array();
    foreach ($sort as $v) {

        $disease_name = getDeseaseName($diseases, $dise[$v]['disease_id']);
        $desease_date = getDeseaseDate($answer['answer_datetime'], $dise[$v]['when_key']);

        $ret .= "{$desease_date}{$disease_name}。\n";
        $done[] = $dise[$v]['disease_id'];
    }
    // 差分(step31_whenに存在しない)症状の処理
    foreach ($done as $disease_id) {
        foreach ($diseases as $key => $val) {
            if ($val['DiseaseID'] == $disease_id) {
                unset($diseases[$key]);
            }
        }
    }
    $no_when = '';
    foreach ($diseases as $val) {
        $no_when .= "{$val['DiseaseName']}。\n";
    }
    $ret = $no_when . $ret;

    return $ret;
}

/**
 * 病名を返却する
 */
function getDeseaseName(Array $diseases, $disease_id)
{
    foreach ($diseases as $v) {
        if ($disease_id == $v['DiseaseID']) return $v['DiseaseName'];
    }
    return '';
}

/*
  症状が出た日を返却する

  例）3月4日から：顔色が悪い

    999 => 'わからない', 
    1   => '今日から', 
    10  => '昨日から', 
    20  => 'おとといから', 
    30  => '3日前から', 
    40  => '4日前から', 
    50  => '5日前から', 
    60  => '6日前から', 
    70  => '1週間前から', 
    80  => '8日前から', 
    90  => '9日前から', 
    100 => '10日前から', 
    110 => '11日前から', 
    120 => '12日前から', 
    130 => '13日前から', 
    140 => '2週間前から', 
    150 => '2週間より前から', 
*/
function getDeseaseDate($datetime, $when_key)
{
    if ((int)$when_key === 999) return 'いつからか分からない ';
    if ((int)$when_key === 1) return date('n月j日 ', strtotime($datetime));

    return date('n月j日 ', strtotime($datetime) - ($when_key / 10) * 86400);
}

/**
 * 外来カルテの履歴を返却する
 */
function getHistorys($cartes)
{
    $ret = array();
    foreach ($cartes as $v) {
        $id = $v['outpatient_carte_id'];
        $reg_datetime = date('y/m/d H:i', strtotime($v['reg_datetime']));
        $doctor_id = $v['doctor_id'];
        $user_div = '';
        foreach (Resource::get('admin.passwd') as $vv) if ($vv['user_id']) {
            $user_div = $vv['user_div'];
            break;
        }

        $ret[$id] = "{$reg_datetime} {$user_div}-{$doctor_id}";
    }

    return $ret;
}

/**
 * 外来カルテのデフォルト値を返却する
 *
 * @param  array $answer 回答
 * @param  array $disease 症状
 * @return array デフォルト値の配列
 *
 * @see 20110908_PEデフォルト選択の補足内容.txt
 */
function getDefault(Array $answer, Array $diseases)
{
    $def = array();

    Factory::import('App/Severity');

    // Chief Complaint
    $def['KARTE100101'] = $answer['primaryShojo'];

    // Onset and Course
    $def['OCContents']  = getOnsetAndCourse($answer, $diseases);

    // Birth Data Weight...
    $def['KARTE180401'] = $answer['kc_q2_text'];

    // Immunization...
    $immunization = '';
    $immuni = array();
    if ($answer['kc_q5']) foreach (explode(',', $answer['kc_q5']) as $v) {
        if ($ret = Resource::get('child.yobou', $v)) {
            if ($ret && $ret !== 'その他') $immuni[] = Resource::get('child.yobou', $v);
        }
    }
    if ($answer['kc_q5_other_text'] !== '') $immuni[] = $answer['kc_q5_other_text'];
    $immunization .= implode('、', $immuni) . "\n";
    if($answer['kc_q5_sanshu']) {
        $immunization .= '・三種混合: ' .  Resource::get('child.sanshu', $answer['kc_q5_sanshu'], '') . "\n";
    }
    if($answer['kc_q5_sanshu']) {
        $immunization .= '・日本脳炎: ' . Resource::get('child.nihon', $answer['kc_q5_nihon'], '') . "\n";
    }
    if($answer['kc_q5_bcg_text']) {
        $immunization .= '・BCG: ' .  $answer['kc_q5_bcg_text'] . "歳\n";
    }
    $def['KARTE200801'] = $immunization;

    // 大人(16歳以上)のアレルギー：自由入力に変更のため不要
    // Allergy...
    //$allergy = Resource::get('yesno', $answer['ka_q17'], '') . ($answer['ka_q17'] == 10 ? (' (' . $answer['ka_q17_text'] . ')') : ''); // Yes(10)の場合はテキストを表示
    //$def['KARTE230301'] = $allergy;

    // Medications...
    $other = $answer['kc_q15_text'];
    $list = explode(',', $answer['kc_q15']);
    $medi = array();
    foreach ((array)$list as $num) {
        if (empty($num)) { continue; }
        if ($ret = Resource::get('child.medicine', $num)) {
            $medi[] = $ret;
        }
    }
    $medication = implode('、', $medi);
    if (in_array(50, $list)) {
        $medication . " ({$other})";
    }
    $def['KARTE240101'] = $medication;

    // Recipe Medicine...(処方箋)上記$listを利用...
    $prefer_medicine = FALSE;
    if (! empty($list)) {
        foreach ($list as $val) {
            switch ($val) {
                case '10' : // 錠剤...
                    $def['medicine_type_2'] = '錠剤';
                    $prefer_medicine = TRUE;
                    break;
                case '20' : // 粉剤...
                    $def['medicine_type_1'] = '粉剤';
                    $prefer_medicine = TRUE;
                    break;
                case '30' : // シロップ...
                    $def['medicine_type_3'] = 'シロップ';
                    $prefer_medicine = TRUE;
                    break;
                default :
            }
        }
    }
    if (! $prefer_medicine) {
        // 好みの薬が無い場合は、粉剤を指定する...
        $def['medicine_type_1'] = '粉剤';
    }

    // Past Illness...
    $illness = array();
    if ($answer['kc_q5']) foreach (explode(',', $answer['kc_q6']) as $v) {
        if ($ret = Resource::get('child.byouki', $v)) {
            if ($ret !== 'その他') $illness[] = Resource::get('child.byouki', $v);
        }
    }
    if ($answer['kc_q6_text'] !== '') $illness[] = $answer['kc_q6_text'];
    $def['TEXT23'] =  implode('、', $illness);

    // Family History...
    /*$hist = 'ご家族で大きな病気をされた方はいらっしゃいますか？ ';
    $hist .= Resource::get('yesno', $answer['kc_q12'], '') . "\n";
    $illness = array();
    if ($answer['kc_q12_family']) foreach (explode(',', $answer['kc_q12_family']) as $v) {
        if ($ret = Resource::get('family', $v)) {
            if ($ret !== 'その他') $illness[] = Resource::get('family', $v);
        }
    }
    if ($answer['kc_q12other_text'] !== '') $illness[] = $answer['kc_q12other_text'];
    $hist .=  '　' . implode('、', $illness);
    $hist .= "\n";
    $hist .= "ご家族でアレルギー（喘息やアトピー含）の方はいますか？";
    $hist .= Resource::get('yesno', $answer['kc_q13'], '') . "\n";
    $allergy = array();
    if ($answer['kc_q13_family']) foreach (explode(',', $answer['kc_q13_family']) as $v) {
        if ($ret = Resource::get('family', $v)) {
            if ($ret !== 'その他') $allergy[] = Resource::get('family', $v);
        }
    }
    if ($answer['kc_q13_other_text'] !== '') $allergy[] = $answer['kc_q13_other_text'];
    $hist .= '　' . implode('、', $allergy);
    $def['TEXT21'] = $hist;*/

    // 体温
    $def['KARTE270101'] = $answer['nowtemp'];

    // 体重
    $def['KARTE280101'] = $answer['weight'];

    /*
    'General condition
    全身状態による重症度、が超重症または重症のとき
    #104#
    全身状態による重症度、が軽症のとき
    #101#
    全身状態による重症度、が中等症のとき
    　症状1、症状2、症状3のいずれかによる重症度判定で、1つでも重症または超重症がある場合
    　#103#
    　上以外の場合
    　#102#
    */
    if (Severity::isOverHigh($answer['zenshin'])) {
        $def['SYMPSUB101'] = 104;
    } elseif (Severity::isLow($answer['zenshin'])) {
        $def['SYMPSUB101'] = 101;
    } elseif (Severity::isMiddle($answer['zenshin'])) {
        if (isset($answer['shojo1']) && Severity::isOverHigh($answer['shojo1'])) {
            $def['SYMPSUB101'] = 103;
        } elseif (isset($answer['shojo2']) && Severity::isOverHigh($answer['shojo2'])) {
            $def['SYMPSUB101'] = 103;
        } elseif (isset($answer['shojo3']) && Severity::isOverHigh($answer['shojo3'])) {
            $def['SYMPSUB101'] = 103;
        } else {
            $def['SYMPSUB101'] = 102;
        }
    }

    /*
    'Activity
    Condition4が3のとき
    #202#
    Condition4が2のとき
    #204#
    上以外のとき
    #205#
    */
    $step2_ans = explode(',', $answer['step2_answers']);
    $condition4 = $step2_ans[3]; // Q.4...
    $revtxt = strrev($condition4);
    $digit = (int)$revtxt[0];
    switch ($digit) {
        case 0 : // 選択肢1(10,25,40,55,70,85,100,115,130)
        case 5 :
            $def['SYMPSUB201'] = 205;
            break;
        case 1 : // 選択肢2(11,26.41,56,71,86,101,116,131)
        case 6 :
            $def['SYMPSUB201'] = 204;
            break;
        case 2 : // 選択肢3(12,27,42,57,72,87,102,117,132)
        case 7 :
            $def['SYMPSUB201'] = 202;
            break;
        default : // 上記以外...
            $def['SYMPSUB201'] = 205;
    }

    /*
    'Mood
    全身状態による重症度、が超重症または重症のとき
    #304#
    全身状態による重症度、が軽症のとき
    #301#
    全身状態による重症度、が中等症のとき
    　症状1、症状2、症状3のいずれかによる重症度判定で、1つでも重症または超重症がある場合
    　#303#
    　上以外の場合
    　#302#
    */
    if (Severity::isOverHigh($answer['zenshin'])) {
        $def['SYMPSUB301'] = 104;
    } elseif (Severity::isLow($answer['zenshin'])) {
        $def['SYMPSUB301'] = 101;
    } elseif (Severity::isMiddle($answer['zenshin'])) {
        if (isset($answer['shojo1']) && Severity::isOverHigh($answer['shojo1'])) {
            $def['SYMPSUB301'] = 103;
        } elseif (isset($answer['shojo2']) && Severity::isOverHigh($answer['shojo2'])) {
            $def['SYMPSUB301'] = 103;
        } elseif (isset($answer['shojo3']) && Severity::isOverHigh($answer['shojo3'])) {
            $def['SYMPSUB301'] = 103;
        } else {
            $def['SYMPSUB301'] = 102;
        }
    }

    /*
    'Sucking(Appetiteと同じ)
    Condition5が3のとき
    #401#
    Condition5が2のとき
    #403#
    上以外のとき
    #404#
    */
    $condition5 = $step2_ans[4]; // Q.5...
    $revtxt = strrev($condition5);
    $digit = (int)$revtxt[0];
    switch ($digit) {
        case 3 : // 選択肢1(13,28,43,58,73,88,103,118,133)
        case 8 :
            $def['SYMPSUB401'] = 404;
            break;
        case 4 : // 選択肢2(14,29.44,59,74,89,104,119,134)
        case 9 :
            $def['SYMPSUB401'] = 403;
            break;
        case 5 : // 選択肢3(15,30,45,60,75,90,105,120,135)
        case 0 :
            $def['SYMPSUB401'] = 401;
            break;
        default : // 上記以外...
            $def['SYMPSUB401'] = 404;
    }

    /*
    'Water intake
    Condition5が3のとき
    #501#
    Condition5が2のとき
    #503#
    上以外のとき
    #504#
    */
    $condition5 = $step2_ans[4]; // Q.5...
    $revtxt = strrev($condition5);
    $digit = (int)$revtxt[0];
    switch ($digit) {
        case 3 : // 選択肢1(13,28,43,58,73,88,103,118,133)
        case 8 :
            $def['SYMPSUB501'] = 504;
            break;
        case 4 : // 選択肢2(14,29.44,59,74,89,104,119,134)
        case 9 :
            $def['SYMPSUB501'] = 503;
            break;
        case 5 : // 選択肢3(15,30,45,60,75,90,105,120,135)
        case 0 :
            $def['SYMPSUB501'] = 501;
            break;
        default : // 上記以外...
            $def['SYMPSUB501'] = 504;
    }


    /*
    'Final stool
    全ての場合で
    #601#
    */
    $def['SYMPSUB601'] = 601;

    /*
    'Urinary output
    全ての場合で
    #701#
    */
    $def['SYMPSUB701'] = 701;

    /*
    'Face color
    全ての場合で
    #801#
    */
    $def['SYMPSUB801'] = 801;

    /*
    'Head
    全ての場合で
    #901#905#
    */
    $def['SYMPSUB2701'] = 901;
    $def['SYMPSUB2702'] = 905;

    /*
    'Eyes & Conjunctiva
    全ての場合で
    #1001#
    */
    $def['SYMPSUB1101_FX1'] = 1001;

    /*
    'Ear drums
    全ての場合で
    #1101#1109#
    */
    $def['SYMPSUB1201_FX1'] = 1101;
    $def['SYMPSUB1202_FX1'] = 1109;

    /*
    'Nose
    全ての場合で
    #1201#
    */
    $def['SYMPSUB1301_FX1'] = 1201;

    /*
    'Throat
    全ての場合で
    #1301#1303#
    */
    $def['SYMPSUB1401_FX1'] = 1301;
    $def['SYMPSUB1402_FX1'] = 1303;

    /*
    'Cervical Lymphnodes
    全ての場合で
    #1401#1407#1409#1410#1412#1418#1420#1421#
    */
    $def['SYMPSUB1501'] = 1401;
    $def['SYMPSUB1502'] = 1407;
    $def['SYMPSUB1503'] = 1409;
    $def['SYMPSUB1504'] = 1410;
    $def['SYMPSUB1505'] = 1412;
    $def['SYMPSUB1506'] = 1418;
    $def['SYMPSUB1507'] = 1420;
    $def['SYMPSUB1508'] = 1421;

    /*
    'Parotid gland
    全ての場合で
    #1501#1507#1509#1510#1512#1518#1520#1521#
    */
    $def['SYMPSUB2601'] = 1501;
    $def['SYMPSUB2602'] = 1507;
    $def['SYMPSUB2603'] = 1509;
    $def['SYMPSUB2604'] = 1510;
    $def['SYMPSUB2605'] = 1512;
    $def['SYMPSUB2606'] = 1518;
    $def['SYMPSUB2607'] = 1520;
    $def['SYMPSUB2608'] = 1521;

    /*
    'Neck stiffness
    全ての場合で
    #1603#
    */
    $def['SYMPSUB901'] = 1603;

    /*
    'Kernig sign
    全ての場合で
    #1703#
    */
    $def['SYMPSUB1001'] = 1703;

    /*
    'Lungs
    全ての場合で
    #1801#1806#1809#
    */
    $def['SYMPSUB1601'] = 1801;
    $def['SYMPSUB1602'] = 1806;
    $def['SYMPSUB1603'] = 1809;

    /*
    'Heart
    全ての場合で
    #1901#1903#1906#1911#1913#
    */
    $def['SYMPSUB1701'] = 1901;
    $def['SYMPSUB1702'] = 1903;
    $def['SYMPSUB1703'] = 1906;
    $def['SYMPSUB1704'] = 1911;
    $def['SYMPSUB1705'] = 1913;

    /*
    'Thorax deformities
    全ての場合で
    #2001#
    */
    $def['SYMPSUB2501'] = 2001;

    /*
    'Abdomen
    全ての場合で
    #2101#2107#2113#2115#2117#2119#2121#2123#2125#2127#2129#2130#2141#
    年齢が1歳より大きいのとき、追加で
    #2103#
    年齢が1歳以下のとき、追加で
    #2104#
    */
    $def['SYMPSUB1801'] = 2101;
    $def['SYMPSUB1803'] = 2107;
    $def['SYMPSUB1804'] = 2113;
    $def['SYMPSUB1805'] = 2115;
    $def['SYMPSUB1806'] = 2117;
    $def['SYMPSUB1807'] = 2119;
    $def['SYMPSUB1808'] = 2121;
    $def['SYMPSUB1809'] = 2123;
    $def['SYMPSUB1810'] = 2125;
    $def['SYMPSUB1811'] = 2127;
    $def['SYMPSUB1812'] = 2129;
    $def['SYMPSUB1813'] = 2130;
    $def['SYMPSUB1814'] = 2141;
    if ($answer['age'] > 1)  $def['SYMPSUB1802'] = 2103;
    if ($answer['age'] <= 1) $def['SYMPSUB1802'] = 2104;

    /*
    'Back
    全ての場合で
    #2202#2204#
    */
    $def['SYMPSUB1901'] = 2202;
    $def['SYMPSUB1902'] = 2204;

    /*
    'Skin
    全ての場合で
    #2301#2302#
    */
    $def['SYMPSUB2001_FX1'] = 2301;
    $def['SYMPSUB2001_FX2'] = 2302;

    /*
    'Extremities
    全ての場合で
    #2401#
    */
    $def['SYMPSUB2101_FX1'] = 2401;

    /*
    'Joints
    全ての場合で
    #2501#
    */
    $def['SYMPSUB2201_FX1'] = 2501;

    /*
    'Genitalia
    全ての場合で
    #2609#2618#2620#2624#2630#
    性別が男のとき、追加で
    #2601#
    性別が女のとき、追加で
    #2602#
    */
    $def['SYMPSUB2303'] = 2609;
    $def['SYMPSUB2305'] = 2618;
    $def['SYMPSUB2306'] = 2620;
    $def['SYMPSUB2307'] = 2624;
    $def['SYMPSUB2309_FX2'] = 2630;
    if ($answer['gender_div_id'] == 1) $def['SYMPSUB2301'] = 2601;
    if ($answer['gender_div_id'] == 2) $def['SYMPSUB2301'] = 2602;

    /*
    'Neurological findings
    全ての場合で
    #9990#9993#
    */
    $def['SYMPSUB9990'] = 9990;
    $def['SYMPSUB9993'] = 9993;

    /*
    Diagnostic Plan と Therapeutic Plan のデフォルトを「なし」に設定する
    */
    $def['RECEPT2'] = '検査なし';
    $def['RECEPT21'] = '処置なし';
 
    /*
    Diagnostic Plan > 画像 > Xp / CT のデフォルト値
    */
    $def['XpOrientation'] = 'Xp正側';
    $def['CTOrientation'] = 'CT単純';
   
    return $def;
}
//getDefaultJson($_answer_code);

function getVitalSign($answer)
{
    $arr = array();

    // 体温
    $arr['KARTE270101'] = $answer['body_temperature'];
    // 脈拍
    $arr['KARTE270201'] = $answer['heart_rate'];
    // 呼吸数
    $arr['KARTE270301'] = $answer['respiratory_rate'];
    // SpO2
    $arr['KARTE271101'] = $answer['spo2'];
    // O2
    $arr['KARTE271201'] = $answer['o2'];
    // 血圧(低)
    $arr['KARTE270401'] = $answer['blood_pressure_low'];
    // 血圧(高)
    $arr['KARTE270501'] = $answer['blood_pressure_high'];
    // 血圧(左)(低)
    $arr['KARTE270601'] = $answer['blood_pressure_l_low'];
    // 血圧(左)(高)
    $arr['KARTE270701'] = $answer['blood_pressure_l_high'];
    // 血圧(左)(低)
    $arr['KARTE270801'] = $answer['blood_pressure_r_low'];
    // 血圧(右)(高)
    $arr['KARTE270901'] = $answer['blood_pressure_r_high'];
    // room air フラグ
    if (isset($answer['room_air_flag']) && $answer['room_air_flag'] == 1) {
        $arr['KARTE271301_FX1'] = 'room air';
    }
    // O2フラグ
    if (isset($answer['o2_flag']) && $answer['o2_flag'] == 1) {
        $arr['KARTE271301_FX2'] = 'O2';
    }
    // 左右別フラグ
    if (isset($answer['blood_pressure_lr_flag']) && $answer['blood_pressure_lr_flag'] == 1) {
        $arr['KARTE271001'] = '左右別';
    }
    // 検査フラグ
    if (isset($answer['dx_flag'])) {
        switch (TRUE) {
            case ($answer['dx_flag'] === '1') :
                $arr['RECEPT2'] = '検査あり';
                break;
            case ($answer['dx_flag'] === '0') :
                $arr['RECEPT2'] = '検査なし';
                break;
            default :
        }
    }
    // 処置フラグ
    if (isset($answer['tx_flag'])) {
        switch (TRUE) {
            case ($answer['tx_flag'] === '1') :
                $arr['RECEPT21'] = '処置あり';
                break;
            case ($answer['tx_flag'] === '0') :
                $arr['RECEPT21'] = '処置なし';
                break;
            default :
        }
    }
    // Assessmentフラグ
    if (isset($answer['assessment_flag'])) {
        switch (TRUE) {
            case ($answer['assessment_flag'] === '1') :
                $arr['RECEPT7'] = '帰宅';
                break;
            case ($answer['assessment_flag'] === '2') :
                $arr['RECEPT7'] = '処置検査後診察';
                break;
            case ($answer['assessment_flag'] === '3') :
                $arr['RECEPT7'] = '入院';
                break;
            default :
        }
    }

    return $arr;
}

