<?php
/**
 * リソース Class
 *
 * @package Lambda/component
 * @author  rooth
 * @version 0.0.1
 *
 * PHP version 5
 *
 *<pre>
 *
 * usage:
 *  Resource::get(string filename [, mixed keys, mixed fallback, bool sanitize]);
 *
 *   #リソースをロードするベースディレクトリを「/path/to/lib」にセット
 *   Resource::setBaseDir('/path/to/lib');
 *   // ※. このメソッドを呼ばない場合、デフォルトではこのクラスが置かれているディレクトリが設定されます。
 *
 *   # filename.php の値を返却する
 *   $rsc = Resource::get('filename');
 *
 *   # filename.php の キー ['key1'] の値を返却する： array('key1' => 'val_1') // val_1
 *   $rsc = Resource::get(filename, 'key1');
 *
 *   # filename.php の キー ['key1']['key2'] の値を返却する： array('key1' => array('key2' => 'val_2')) // VAL_2
 *   $rsc = Resource::get(filename, array('key1', 'key2'));
 *
 *   # fallback: 存在しない時の代替
 *   $rsc = Resource::get(filename, array('key1', 'key2'), '');
 *
 *   # 値をサニタイズしないで返却する
 *   $rsc = Resource::get(filename, null, array(), Resource::INSANITATION);
 *
 *   # 指定可能なキーの深さは「5」まで。
 *
 *      # 動作します
 *      $rsc = Resource::get('filename', array('key1', 'key2', 'key3', 'key4', 'key5'));
 *
 *      # 動作しません
 *      $rsc = Resource::get('filename', array('key1', 'key2', 'key3', 'key4', 'key5', 'key6'));
 *
 *</pre>
 */

defined('DS') || define('DS', DIRECTORY_SEPARATOR);

class Resource
{
    protected static $charset = 'UTF-8';

    protected static $base_dir;
    protected static $resource = array();

    const SANITATION   = true;
    const INSANITATION = false;

    /**
     * ベースディレクトリをセットする
     *
     * @param  str  $dir ベースディレクトリ
     * @access public
     * @static
     */
    public static function setBaseDir($dir)
    {
        self::$base_dir = (substr($dir, -1) === DS) ? rtirm($dir, DS) : $dir;
    }

    /**
     * ベースディレクトリを返却する
     *
     * @return str ベースディレクトリ
     * @access protected
     * @static
     */
    protected static function getBaseDir()
    {
        if (self::$base_dir === null) self::setBaseDir(dirname(__FILE__));
        return self::$base_dir;
    }

    /**
     * リソースを返却する
     *
     * @param str $file リソースファイル(.php は省略する)
     * @param mixed $keys 取得したいキー名
     *                      null:            指定なし(すべてを取得する)
     *                      str, int, float: 第１レベル
     *                      array:           第２～第５レベル
     * @para mixed $fallback 存在しない時の代替
     * @param bool $sanitize 値をサニタイズするか否か
     * @return mixed 存在する時:mixed / 存在しない時:空の配列
     * @access public
     * @static
     */
    public static function get($file, $keys = null, $fallback = array(), $sanitize = self::SANITATION)
    {
        self::setResource($file);
        if ( ! $rsc = self::getResource($file)) return $fallback;

        if ($sanitize) $rsc = self::rh($rsc);

        if ($keys === null) {
            if (is_array($fallback) && empty($fallback)) {
                return $rsc;
            } else {
                return $fallback;
            }
        }
        if ( ! is_array($keys) && isset($rsc[$keys])) return $rsc[$keys];
        if ( ! is_array($keys) && ! isset($rsc[$keys])) return $fallback;

        $k =& $keys;
        switch (count($k)) {
            case 1:
                return isset($rsc[$k[0]]) ? $rsc[$k[0]] : $fallback;
            case 2:
                return isset($rsc[$k[0]][$k[1]]) ? $rsc[$k[0]][$k[1]] : $fallback;
            case 3:
                return isset($rsc[$k[0]][$k[1]][$k[2]]) ? $rsc[$k[0]][$k[1]][$k[2]] : $fallback;
            case 4:
                return isset($rsc[$k[0]][$k[1]][$k[2]][$k[3]]) ? $rsc[$k[0]][$k[1]][$k[2]][$k[3]] : $fallback;
            case 5:
                return isset($rsc[$k[0]][$k[1]][$k[2]][$k[3]][$k[4]]) ? $rsc[$k[0]][$k[1]][$k[2]][$k[3]][$k[4]] : $fallback;
            default:
                return $fallback;
        }
    }



    /**
     * リソースをセットする
     *
     * @param str $file リソースファイル
     * @access protected
     * @static
     */
    protected static function setResource($file)
    {
        if (isset(self::$resource[$file])) return;

        $path = self::getBaseDir().DS.$file.'.php';
        if ( ! is_readable($path)) return;

        self::$resource[$file] = require $path;
    }

    /**
     * リソースを返却する
     *
     * @param str $file リソースファイル
     * @return array リソースの配列
     * @access protected
     * @static
     */
    protected static function getResource($file)
    {
        return isset(self::$resource[$file]) ? self::$resource[$file] : array();
    }

    /**
     * recursive htmlspecialchars
     *
     * @access protected
     * @static
     */
    protected static function rh($mixed)
    {
        if (is_scalar($mixed) && ! is_bool($mixed)) {
            return htmlspecialchars($mixed, ENT_QUOTES, self::$charset);
        } elseif (is_array($mixed)) {
            return array_map(array('self', 'rh'), $mixed);
        } else {
            return $mixed;
        }
    }



} // -- End of class


