<?php
/**
 * 重症度 class
 */

/*
----------------------------------------------------------------------
・重症度
・重症化リスク(緊急度)
----------------------------------------------------------------------
重症度は、今の状態の指標。超重症/重症/中等症/軽症。
重症化リスクは、今後悪化の可能性。高/中/低。
または新しい出力形式（緊急度区分表示）として、蘇生/緊急/準緊急/非緊急。

なお、判定結果は下記のように数値で管理される。

　16歳未満
　超重症(=6)・重症(=5)・中等症(=4)・軽症(=2)

　16歳以上
　超重症(=5)・重症(=4)・中等症(=3)・軽症(=1)

ただし、内部では平坦化して扱いやすくするため、統一する。
具体的には、16歳以上未満ともに下記となる。

　超重症(=4)・重症(=3)・中等症(=2)・軽症(=1)


----------------------------------------------------------------------
・Base重症度 (base_severity)
・Custom重症度 (custom_severity)
・Base重症化リスク (base_risk)
・Custom重症化リスク (custom_risk)
----------------------------------------------------------------------
重症度/重症化リスクには、判定結果に何も手を加えない基礎値がある。
この基礎値を便宜上、「Base重症度 / Base重症化リスク」と呼称する。

また、基礎値をロール(医師、小児科医、看護師、事務など)によってフィルタリング(基礎値の増減)を施したカスタム値が存在する。
このカスタム値を便宜上、「Custom重症度 / Custom重症化リスク」と呼称する。

そのため、重症度判定結果に対し、医師が利用する「Custom重症度」と、事務が利用する「Custom重症度」の値が異なることがある。


----------------------------------------------------------------------
重症度の求め方(16歳未満)
----------------------------------------------------------------------
判断要素は、最大で
① 全身状態 （5問）から求められる重症度判定
② 外傷選択1（3問）から求められる重症度判定
③ 外傷選択2（3問）から求められる重症度判定
④ 外傷選択3（3問）から求められる重症度判定
⑤ 発熱     （3問）から求められる重症度判定
⑥ 症状選択1（3問）から求められる重症度判定
⑦ 症状選択2（3問）から求められる重症度判定
⑧ 症状選択3（3問）から求められる重症度判定
の以上8つとなる。
これらについては、項目毎にそれぞれ設問があるので、それを用いてそれぞれ重症度を求める事が出来る。
入力が無い項目については、判断要素に含めない。
判断要素を全て求めた後、その中で一番重い判断結果を、最終的な判断結果＝「Base重症度」として採用する。

ユーザロール(医師、小児科医、看護師、事務など)によって、「Base重症度」を調整した値＝「Custom重症度」をシステムでは画面表示等に利用する。
ユーザロールは個別に持たせるが、まずは「小児科医」の場合のみ下記を適応する。

主訴がその他以外の場合
 ┗ ①が軽症の場合（最初に処理）
    ┗ 最終的な判断結果が超重症の場合・・・重症に下げる
    ┗ 最終的な判断結果が重症の場合  ・・・中等症に下げる
 ┗ 性年齢カテゴリが1又は11の場合（次に処理）
    ┗ 現在体温が38度以上の場合          ・・・重症以上とする
    ┗ 現在体温が37.5度以上38度未満の場合・・・中等症以上とする
主訴がその他の場合
 ┗ 最終的な判断結果が中等症の場合・・・重症に上げる
 ┗ 最終的な判断結果が軽症の場合  ・・・中等症に上げる


----------------------------------------------------------------------
重症化リスクの求め方(16歳未満)
----------------------------------------------------------------------
「Base重症度」から、基礎値としての「Base重症化リスク」を求める。
 超重症の場合・・・高
 重症  の場合・・・高
 中等症の場合・・・中
 軽症  の場合・・・低

ユーザロールによって、「Base重症化リスク」を調整した値＝「Custom重症化リスク」とする。
ユーザロールは個別に持たせるが、まずは「小児科医」の場合のみ下記を適応する。

①が中等症の場合
 ┗ ②③④で最も重いものが重症の場合・・・緊急度を高から中に下げる


----------------------------------------------------------------------
Base重症度の求め方(16歳以上)
----------------------------------------------------------------------
判断要素は、最大で
① 全身状態 （5問）から求められる重症度判定
② 外傷選択1（3問）から求められる重症度判定
③ 外傷選択2（3問）から求められる重症度判定
④ 外傷選択3（3問）から求められる重症度判定
⑤ 症状選択1（3問）から求められる重症度判定
⑥ 症状選択2（3問）から求められる重症度判定
⑦ 症状選択3（3問）から求められる重症度判定
の以上7つとなる。
これらについては、項目毎にそれぞれ設問があるので、それを用いてそれぞれ重症度を求める事が出来る。
入力が無い項目については、判断要素に含めない。（16歳未満と違い、⑤⑥⑦で入力があっても判定が無い事がある）
判断要素を全て求めた後、その中で一番重い判断結果を、最終的な判断結果＝「Base重症度」として採用する。

ユーザロールによって、「Base重症度」を調整した値＝「Custom重症度」をシステムでは画面表示等に利用する。
ユーザロールは個別に持たせるが、まずは「医師」「小児科医」の場合下記を適応する。

主訴がその他以外の場合
 ┗ ①が軽症の場合（最初に処理）
    ┗ 最終的な判断結果が超重症の場合・・・重症に下げる
    ┗ 最終的な判断結果が重症の場合  ・・・中等症に下げる
主訴がその他の場合
 ┗ 最終的な判断結果が中等症の場合・・・重症に上げる
 ┗ 最終的な判断結果が軽症の場合  ・・・中等症に上げる


----------------------------------------------------------------------
重症化リスクの求め方(16歳以上)
----------------------------------------------------------------------
「Base重症度」から、基礎値としての「Base重症化リスク」を求める。
 超重症の場合・・・高
 重症  の場合・・・高
 中等症の場合・・・中
 軽症  の場合・・・低

ユーザロールによって、「Base重症化リスク」を調整した値＝「Custom重症化リスク」とする。
ユーザロールは個別に持たせるが、まずは「医師」「小児科医」の場合下記を適応する。

①が中等症の場合
 ┗ ②③④で最も重いものが重症の場合・・・緊急度を高から中に下げる


以上。

*/



class Severity
{
    const CHILD = 1; // 16歳未満
    const ADULT = 2; // 16歳以上

    private $mode;

    private $dao;
    private $zenshin_category_id;
    private $shojo_category_id;

    private $severities = array(); // 重症度配列
    private $diagnosis = '';
    private $shojo_severities = array();
    private $gaisho_severities = array();
    private $fever_diagnosis = array();

    private $exec_flag = false;

    private $severity_level, // 重症度(返却用)
            $risk_level,     // 重症化リスク(返却用)
            $serious_level;  // シリアスレベル(返却用)

    private $level_superhigh,  // 超重症
            $level_high,       // 重症
            $level_middle,     // 中等症
            $level_low;        // 軽症

    const LEVEL6 = 6;
    const LEVEL5 = 5;
    const LEVEL4 = 4;
    const LEVEL3 = 3;
    const LEVEL2 = 2;
    const LEVEL1 = 1;

    const LEVEL_SUPERHIGH = self::LEVEL4;
    const LEVEL_HIGH      = self::LEVEL3;
    const LEVEL_MIDDLE    = self::LEVEL2;
    const LEVEL_LOW       = self::LEVEL1;

    const LABEL_SEVERITY_SUPERHIGH = SEVERITY_LABEL_SUPERHIGH;
    const LABEL_SEVERITY_HIGH      = SEVERITY_LABEL_HIGH;
    const LABEL_SEVERITY_MIDDLE    = SEVERITY_LABEL_MIDDLE;
    const LABEL_SEVERITY_LOW       = SEVERITY_LABEL_LOW;

    // 緊急度は、今後悪化の可能性。高/中/低。
    // または新しい出力形式（緊急度区分表示）として、蘇生/緊急/準緊急/非緊急。
    const LABEL_RISK_SUPERHIGH = '高';
    const LABEL_RISK_HIGH      = '高';
    const LABEL_RISK_MIDDLE    = '中';
    const LABEL_RISK_LOW       = '低';



    public function __construct($mode = self::CHILD)
    {
        $this->mode = $mode;

        if ($this->mode == self::CHILD) {
            /*
            16歳未満
              超重症(=6)・重症(=5)・中等症(=4)・軽症(=2)
             */
            $this->level_superhigh = self::LEVEL6;
            $this->level_high      = self::LEVEL5;
            $this->level_middle    = self::LEVEL4;
            $this->level_low       = self::LEVEL2;
        }
        if ($this->mode == self::ADULT) {
            /*
            16歳以上
              超重症(=5)・重症(=4)・中等症(=3)・軽症(=1)
             */
            $this->level_superhigh = self::LEVEL5;
            $this->level_high      = self::LEVEL4;
            $this->level_middle    = self::LEVEL3;
            $this->level_low       = self::LEVEL1;
        }
    }

    public function prepare($zenshin_category_id, $shojo_category_id)
    {
        if ($this->mode == self::CHILD) {
            $this->dao = Factory::load('DB/DAO_monshin_child')->getInstance();
        }
        if ($this->mode == self::ADULT) {
            $this->dao = Factory::load('DB/DAO_monshin_adult')->getInstance();
        }
        $this->zenshin_category_id = $zenshin_category_id;
        $this->shojo_category_id   = $shojo_category_id;
    }

    /**
     * Base重症度(base_severity) の計算に必要な処理を実行する
     */
    public function execute(Array $step2_answers, Array $step31_answers, Array $step212_answers, Array $fever_answers)
    {
        $this->zenshinSeverity($step2_answers);
        $this->shojoSeverity($step31_answers);
        if ($this->mode == self::CHILD) $this->gaishoSeverity($step212_answers);
        if ($this->mode == self::CHILD) $this->feverDiagnosis($fever_answers); // 16歳未満のみ、発熱を考慮する

        $this->exec_flag = true;
    }

    /**
     * 症状情報を返却する
     */
    public function getSeverities()
    {
        if ( ! $this->exec_flag) {
            exit('getSeverities: executeメソッドを実行してからこのメソッドを実行してください');
        }

        $ret = array();
        foreach ($this->severities as $k => $v) {
            $ret[$k] = $this->_adjustLevel($v);
        }

        return $ret;
    }

    /**
     * Base重症度(base_severity)
     *
     * @return integer
     * @see doc/いただいたドキュメント/重症度／緊急度(重症化リスク)/算出指標{1,2}.txt
     */
    public function getBase()
    {
        if ( ! $this->exec_flag) {
            exit('getBase: executeメソッドを実行してからこのメソッドを実行してください');
        }

        $buf = array();
        foreach ($this->severities as $k => $v) {
            $buf[$k] = $this->_adjustLevel($v);
        }

        return max($buf);
    }

    /**
     * Custom重症度(custom_severity)
     *
     * @param integer $adjusted_base_severity 調整済みBase重症度
     * @oaram integer $adjusted_zenshin_severity 調整済み全身状態の重症度
     * @param string $primary_shojo 主訴
     * @oaram float $nowtemp 現在の体温
     * @return integer
     *
     * @see doc/いただいたドキュメント/重症度／緊急度(重症化リスク)/算出指標{1,2}.txt
     */
    public function getCustom($adjusted_base_severity, $adjusted_zenshin_severity, $primary_shojo, $nowtemp)
    {
        $level = null;

        if ($this->mode == self::CHILD) {
            // 重症度2(16歳未満)
            $level = $this->_getChildCustomSeverity(
                $adjusted_base_severity, 
                $adjusted_zenshin_severity, 
                $primary_shojo, 
                $nowtemp
            );
        }

        if ($this->mode == self::ADULT) {
            // 重症度2(16歳以上)
            $level = $this->_getAdultCustomSeverity(
                $adjusted_base_severity, 
                $adjusted_zenshin_severity, 
                $primary_shojo
            );
        }

        return ($level === null) ? null : (float)$level;
    }

    /**
     * Base重症化リスク(base_risk)
     *
     * @param integer $adjusted_base_severity 調整済みBase重症度
     * @return integer
     * @see doc/いただいたドキュメント/重症度／緊急度(重症化リスク)/算出指標{1,2}.txt
     */
    public function getBaseRisk($adjusted_base_severity)
    {
        /*
        重症化リスクの求め方

        「Base重症度」から、基礎値としての「Base重症化リスク」を求める。
         超重症の場合・・・高
         重症  の場合・・・高
         中等症の場合・・・中
         軽症  の場合・・・低
        */

        return $adjusted_base_severity;
    }

    /**
     * Cusotm重症化リスク(custom_risk)
     *
     * @param integer $base_severity 調整済みBase重症度
     * @oaram integer $adjusted_zenshin_severity 調整済み全身状態の重症度
     * @oaram array $adjusted_gaisho_severities 調整済み外傷の重症度
     * @return integer
     * @see doc/いただいたドキュメント/重症度／緊急度(重症化リスク)/算出指標{1,2}.txt
     */
    public function getCustomRisk($adjusted_base_severity, $adjusted_zenshin_severity, Array $adjusted_gaisho_severities)
    {
        /*
        重症化リスクの求め方

        「Base重症度」から、基礎値としての「Base重症化リスク」を求める。
         超重症の場合・・・高
         重症  の場合・・・高
         中等症の場合・・・中
         軽症  の場合・・・低

        ユーザロールによって、「Base重症化リスク」を調整した値＝「Custom重症化リスク」とする。
        ユーザロールは個別に持たせるが、まずは「小児科医」の場合のみ下記を適応する。

        ①が中等症の場合
         ┗ ②③④で最も重いものが重症の場合・・・緊急度を高から中に下げる

            [①～④の定義]
            ① 全身状態 （5問）から求められる重症度判定
            ② 外傷選択1（3問）から求められる重症度判定
            ③ 外傷選択2（3問）から求められる重症度判定
            ④ 外傷選択3（3問）から求められる重症度判定
        */

        $level = $adjusted_base_severity;
        if ($adjusted_zenshin_severity == self::LEVEL_MIDDLE) {
            $max = max($adjusted_gaisho_severities);
            if ($max == self::LEVEL_HIGH) {
                $level = self::LEVEL_MIDDLE;
            }
        }

        return (float)$level;
    }



    /**
     * 重症度/重症化リスクを判定する
     *
     * @param  integer answer_id 回答ID
     * @param  Severity severity Severityインスタンス
     * @param  boolian isRoleDoctor 医師のロールを持っているか否か
     * @return array array('重症度' ,'重症化リスク')
    */
    public function judge($answer_id, Severity $severity, $isRoleDoctor)
    {
        $sev_info = Factory::load('DB/DAO_severitys')->getInstance()->fetch_LEFT_JOIN_answers_BY_answer_id($answer_id);
        $zenshin_category_id = $sev_info['zenshin_category_id'];
        $shojo_category_id   = $sev_info['shojo_category_id'];
        $step2_answers   = explode(',', $sev_info['step2_answers']);
        $step31_answers  = explode(',', $sev_info['step31_answers']);
        $step212_answers = explode(',', $sev_info['step212_answers']);
        $fever_answers   = explode(',', $sev_info['fever_answers']);
        $primary_shojo   = $sev_info['primaryShojo'];
        $nowtemp         = $sev_info['nowtemp'];

        $gaisho_sevs = array($sev_info['gaisho1'], $sev_info['gaisho2'], $sev_info['gaisho3']);
        $zenshin     = $sev_info['zenshin'];

        $severity->prepare($zenshin_category_id, $shojo_category_id);
        $severity->execute($step2_answers, $step31_answers, $step212_answers, $fever_answers);

        if ($isRoleDoctor) {
            if ($sev_info['final_severity'] === null) {
                $this->severity_level = $severity->getCustom($sev_info['base_severity'], $zenshin, $primary_shojo, $nowtemp);
            } else {
                $this->severity_level = $sev_info['final_severity'];
            }
            $this->risk_level = $severity->getCustomRisk(
                $this->severity_level, 
                $zenshin, 
                $gaisho_sevs
            );
        } else {
            // final_severity: 重症度が修正された場合、修正された方を優先する
            $final_severity = ($sev_info['final_severity'] === null) ? $sev_info['base_severity'] : $sev_info['final_severity'];

            $this->severity_level = $final_severity;
            $this->risk_level = $severity->getBaseRisk($final_severity);
        }

        $this->serious_level = $sev_info['serious_level'];

    }
    public function getSeverityLevel()
    {
        return $this->severity_level;
    }
    public function getRiskLevel()
    {
        return $this->risk_level;
    }
    public function getSeriousLevel()
    {
        return $this->serious_level;
    }





    /**
     * 全身状態の重症度判定結果を抽出
     */
    public function zenshinSeverity(Array $step2_answers)
    {
                                                                     // Child     | Adult
        $param1 = isset($step2_answers[0]) ? $step2_answers[0] : ''; // injury_id | consciousness_id
        $param2 = isset($step2_answers[1]) ? $step2_answers[1] : ''; // consciousness_id | breath_id
        $param3 = isset($step2_answers[2]) ? $step2_answers[2] : ''; // breath_id | dialog_id
        $param4 = isset($step2_answers[3]) ? $step2_answers[3] : ''; // activity_id | activity_id
        $param5 = isset($step2_answers[4]) ? $step2_answers[4] : ''; // dehydration_id | dehydration_id
        $param6 = $this->zenshin_category_id;

        $res = $this->dao->step6_getZenshinSeverity($param1, $param2, $param3, $param4, $param5, $param6);
        //_Ve($res);
        $this->diagnosis = $res['DiagnosisContents'];
        //_ve($this->diagnosis);
        $this->severities['zenshin'] = $res['DiagnosisID'];
    }

    public function getSeveritysDataByName($name)
    {
        $result = isset($this->severities[$name]) ? $this->severities[$name] : FALSE;
        return $this->_adjustLevel($result);
    }

    public function getDiagnosis()
    {
        return $this->diagnosis;
    }



    /**
     * 症状選択1の重症度判定結果を抽出
     */
    private function shojoSeverity(Array $step31_answers)
    {
        if ($step31_answers) {
            foreach (array_chunk($step31_answers, 3) as $answer_ids) {
                $this->shojo_severities[] = $this->dao->step6_getShojoSeverity($answer_ids, $this->shojo_category_id);
            }
        }
        //_ve($this->shojo_severities);
        $this->severities['shojo1'] = isset($this->shojo_severities[0]['DiagnosisID']) ? $this->shojo_severities[0]['DiagnosisID'] : '';
        $this->severities['shojo2'] = isset($this->shojo_severities[1]['DiagnosisID']) ? $this->shojo_severities[1]['DiagnosisID'] : '';
        $this->severities['shojo3'] = isset($this->shojo_severities[2]['DiagnosisID']) ? $this->shojo_severities[2]['DiagnosisID'] : '';
    }

    /**
     * step6: 外傷選択3の重症度判定結果を抽出
     */
    private function gaishoSeverity(Array $step212_answers)
    {
        if ($step212_answers) {
            foreach (array_chunk($step212_answers, 3) as $answer_ids) {
                $this->gaisho_severities[] = $this->dao->step6_getGaishoSeverity($answer_ids);
            }
        }
        //_ve($this->gaisho_severities);
        $this->severities['gaisho1'] = isset($this->gaisho_severities[0]['DiagnosisID']) ? $this->gaisho_severities[0]['DiagnosisID'] : '';
        $this->severities['gaisho2'] = isset($this->gaisho_severities[1]['DiagnosisID']) ? $this->gaisho_severities[1]['DiagnosisID'] : '';
        $this->severities['gaisho3'] = isset($this->gaisho_severities[2]['DiagnosisID']) ? $this->gaisho_severities[2]['DiagnosisID'] : '';
    }

    /**
     * step6: 発熱の重症度判定結果を抽出
     */
    private function feverDiagnosis(Array $fever_answers)
    {
        if ($fever_answers) {
            $this->fever_diagnosis = $this->dao->step6_getFeverSeverity($fever_answers, $this->shojo_category_id);
        }
        //_Ve($this->fever_diagnosis);
        $this->severities['fever'] = isset($this->fever_diagnosis['DiagnosisID']) ? $this->fever_diagnosis['DiagnosisID'] : '';
    }
    public function getfeverDiagnosis()
    {
        return $this->fever_diagnosis;
    }

    /**
     * step6: 全身状態の処置対応を抽出
     *
     * ClassID=1の場合には外傷についての質問
     * ClassID=2の場合には意識状態についての質問
     * ClassID=3の場合には呼吸状態についての質問
     * ClassID=4の場合には活動度についての質問
     * ClassID=5の場合には脱水についての質問
     */
    public function getZenshinDiagnosis(Array $step2_answers)
    {
        $zenshin_diagnosis_arr = array();

        // step2: 回答位置
        $s2_answers_pos = array();
        foreach ($step2_answers as $v) {
            // 回答選択肢の位置が「上, 中, 下」のどれであるかを求める
            $v = $v % 3;
            if ($v === 0) $v = 3;
            $s2_answers_pos[] = $v;
        }

        for ($class_id = 1; $class_id <= 5; $class_id++) {
            $answer_id = $s2_answers_pos[$class_id - 1];
            $zenshin_diagnosis_arr[] = $this->dao->step6_getZenshinDiagnosis($this->zenshin_category_id, $class_id, $answer_id);
        }
        //_Ve($zenshin_diagnosis_arr);
        return $zenshin_diagnosis_arr;
    }

    /**
     * 症状選択の処置対応を抽出
     */
    public function getShojoDiagnosis(Array $step31_answers)
    {
        $shojo_diagnosis_arr = array();
        if ($step31_answers) {
            foreach (array_chunk($step31_answers, 3) as $answer_ids) {
                $ret = $this->dao->step6_getShojoDiagnosis($answer_ids, $this->shojo_category_id);
                if ($ret) $shojo_diagnosis_arr[] = $ret;
            }
        }
        //_ve($shojo_diagnosis_arr);
        return $shojo_diagnosis_arr;
    }

    /**
     * 外傷選択の処置対応を抽出①
     */
    public function getGaishoDiagnosis(Array $step211_answers)
    {
        $gaisho_diagnosis_arr = array();
        if ($step211_answers) {
            foreach ($step211_answers as $injury_id) {
                $ret = $this->dao->step6_getGaishoDiagnosis($injury_id);
                if ($ret) $gaisho_diagnosis_arr[] = $ret;
            }
        }
        //_ve($gaisho_diagnosis_arr);
        return $gaisho_diagnosis_arr;
    }

    /**
     * 外傷選択の処置対応を抽出②
     */
    public function getGaishoDetailDiagnosis(Array $step212_answers)
    {
        $gaisho_detail_diagnosis_arr = array();
        if ($step212_answers) {
            foreach (array_chunk($step212_answers, 3) as $v) {
                $res1_id = isset($v[0]) ? $v[0] : '';
                $res2_id = isset($v[1]) ? $v[1] : '';
                $res3_id = isset($v[2]) ? $v[2] : '';

                $ret = $this->dao->step6_getGaishoDetailDiagnosis($res1_id, $res2_id, $res3_id);
                if ($ret) $gaisho_detail_diagnosis_arr[] = $ret;
            }
        }
        //_ve($gaisho_detail_diagnosis_arr);
        return $gaisho_detail_diagnosis_arr;
    }

    /**
     * 誤飲選択の処置対応を抽出
     */
    public function getGoinDiagnosis(Array $step22_answers)
    {
        $goin_diagnosis_text = '';
        if ($step22_answers) {
            $object_id = $step22_answers[0];
            $res = $this->dao->step6_getGoinDiagnosis($object_id);
            $goin_diagnosis_text = $res['SwallowTreatmentContents'];
        }
        //_ve($goin_diagnosis_text);
        return $goin_diagnosis_text;
    }



    /**
     * 16歳未満のCusotm重症度(custom_severity) を返却する
     *
     * @param integer $adjusted_base_severity 調整済みBase重症度
     * @param integer $zenshin_severity 全身症状重症度
     * @param $primary_shojo 主訴
     * @oaram $nowtemp 現在の体温
     * @return integer
     */
    private function _getChildCustomSeverity($adjusted_base_severity, $zenshin_severity, $primary_shojo, $nowtemp)
    {
        /*
        *******************************************
        IMD_WDK事前問診_機能仕様書v1.0.doc - P.42 -
        *******************************************
        ----ここから----
        主訴がその他以外の場合
        　┗　①が軽症の場合（最初に処理）
        　　　　┗　最終的な判断結果が超重症の場合		・・・重症に下げる
        　　　　┗　最終的な判断結果が重症の場合		・・・中等症に下げる
        　┗　性年齢カテゴリが1又は11の場合（次に処理）
        　　　　┗　現在体温が38度以上の場合			・・・重症以上とする
        　　　　┗　現在体温が37.5度以上38度未満の場合	・・・中等症以上とする
        主訴がその他の場合
        　┗　最終的な判断結果が中等症の場合			・・・重症に上げる
        　┗　最終的な判断結果が軽症の場合			・・・中等症に上げる
        ----ここまで----
        */
        /*
        性年齢カテゴリが1又は11の場合（次に処理）
        についてですが、症状のときに使ったカテゴリです。
        */

        $ret = $adjusted_base_severity;
        if ($primary_shojo !== 'その他') {
            if ($zenshin_severity == self::LEVEL_LOW) {
                if ($adjusted_base_severity == self::LEVEL_SUPERHIGH) {
                    $ret = self::LEVEL_HIGH;
                }
                if ($adjusted_base_severity == self::LEVEL_HIGH) {
                    $ret = self::LEVEL_MIDDLE;
                }
            }

            $nowtemp = (float)$nowtemp;
            if ($this->shojo_category_id == 1 || $this->shojo_category_id == 11) {
                if ($nowtemp >= 38) {
                    $ret = self::LEVEL_HIGH;
                } elseif (37.5 <= $nowtemp && $nowtemp < 38) {
                    $ret = self::LEVEL_MIDDLE;
                }
            }
        } else {
            if ($adjusted_base_severity == self::LEVEL_MIDDLE) {
                $ret = self::LEVEL_HIGH;
            }
            if ($adjusted_base_severity == self::LEVEL_LOW) {
                $ret = self::LEVEL_MIDDLE;
            }
        }

        return $ret;
    }

    /**
     * 16歳以上のCusotm重症度(custom_severity) を返却する
     *
     * @param integer $adjusted_base_severity 調整済みBase重症度
     * @param integer $zenshin_severity 全身症状重症度
     * @param $primary_shojo 主訴
     * @return integer
     */
    private function _getAdultCustomSeverity($adjusted_base_severity, $zenshin_severity, $primary_shojo)
    {
        /*
        *******************************************
        算出指標2.txt
        *******************************************
        ----ここから----
        主訴がその他以外の場合
         ┗ ①が軽症の場合（最初に処理）
            ┗ 最終的な判断結果が超重症の場合・・・重症に下げる
            ┗ 最終的な判断結果が重症の場合  ・・・中等症に下げる
        主訴がその他の場合
         ┗ 最終的な判断結果が中等症の場合・・・重症に上げる
         ┗ 最終的な判断結果が軽症の場合  ・・・中等症に上げる
        ----ここまで----
        */

        $ret = $adjusted_base_severity;
        if ($primary_shojo !== 'その他') {
            if ($zenshin_severity == self::LEVEL_LOW) {
                if ($adjusted_base_severity == $this->level_superhigh) {
                    $ret = self::LEVEL_HIGH;
                }
                if ($adjusted_base_severity == self::LEVEL_HIGH) {
                    $ret = self::LEVEL_MIDDLE;
                }
            }
        } else {
            if ($adjusted_base_severity == self::LEVEL_MIDDLE) {
                $ret = self::LEVEL_HIGH;
            }
            if ($adjusted_base_severity == self::LEVEL_LOW) {
                $ret = self::LEVEL_MIDDLE;
            }
        }

        return $ret;
    }

    /**
     * レベル値を調整する
     *  ⇒ 救急外傷のレベル値(1,2,3,4)に合わせる
     */
    private function _adjustLevel($level)
    {
        switch (true) {
            case (int)$level === $this->level_superhigh:
                return (float)self::LEVEL4;

            case (int)$level === $this->level_high:
                return (float)self::LEVEL3;

            case (int)$level === $this->level_middle:
                return (float)self::LEVEL2;

            case (int)$level === $this->level_low:
                return (float)self::LEVEL1;

            default:
                // どれにもマッチしない場合は「軽症」として扱う
                return (float)self::LEVEL_LOW;
        }
    }



    // 重症以上
    public static function isOverHigh($v)
    {
        return (bool)((float)$v >= (float)self::LEVEL_SUPERHIGH);
    }

    // 超重症
    public static function isSuperHigh($v)
    {
        return (bool)((float)$v === (float)self::LEVEL_SUPERHIGH);
    }

    // 重症
    public static function isHigh($v)
    {
        return (bool)((float)$v === (float)self::LEVEL_HIGH);
    }

    // 中等症
    public static function isMiddle($v)
    {
        return (bool)((float)$v === (float)self::LEVEL_MIDDLE);
    }

    // 軽症
    public static function isLow($v)
    {
        return (bool)((float)$v === (float)self::LEVEL_LOW);
    }



    /**
     * Get Severity Text
     * @param integer severity : answers.result
     * @return string Severity Text
     */
    public static function getSeverityText($severity)
    {
        $text = '';
        switch ($severity) {
            case self::LEVEL_SUPERHIGH:
                $text = self::LABEL_SEVERITY_SUPERHIGH;
                break;

            case self::LEVEL_HIGH:
                $text = self::LABEL_SEVERITY_HIGH;
                break;

            case self::LEVEL_MIDDLE:
                $text = self::LABEL_SEVERITY_MIDDLE;
                break;

            case self::LEVEL_LOW:
                $text = self::LABEL_SEVERITY_LOW;
                break;

            default :
        }
        return $text;
    }

    /**
     * Get Risk Text
     * @param integer risk : answers.risk
     * @return string Risk Text
     */
    public static function getRiskText($risk)
    {
        $text = '';
        switch ($risk) {
            case self::LEVEL_SUPERHIGH:
                $text = self::LABEL_RISK_SUPERHIGH;
                break;

            case self::LEVEL_HIGH:
                $text = self::LABEL_RISK_HIGH;
                break;

            case self::LEVEL_MIDDLE:
                $text = self::LABEL_RISK_MIDDLE;
                break;

            case self::LEVEL_LOW:
                $text = self::LABEL_RISK_LOW;
                break;

            default :
        }
        return $text;
    }

    /**
     * Get level mater image
     * @param integer $severity : 重症度 1 - 4
     * @param integer $risk     : 重症化リスク 1 - 4
     */
    public static function getLevelImage($severity, $risk)
    {
        switch ($severity) {
            case self::LEVEL_SUPERHIGH :
                $file = 'LEVEL4B.gif';
                break;
            case self::LEVEL_HIGH :
                $file = 'LEVEL3B.gif';
                break;
            case self::LEVEL_MIDDLE :
                switch ($risk) {
                    case self::LEVEL_SUPERHIGH :
                    case self::LEVEL_HIGH :
                        $file = 'LEVEL25B.gif';
                        break;
                    case self::LEVEL_MIDDLE :
                        $file = 'LEVEL225B.gif';
                        break;
                    default :
                        $file = 'LEVEL2B.gif';
                }
                break;
            case self::LEVEL_LOW :
                switch ($risk) {
                    case self::LEVEL_SUPERHIGH :
                    case self::LEVEL_HIGH :
                        $file = 'LEVEL15B.gif';
                        break;
                    case self::LEVEL_MIDDLE :
                        $file = 'LEVEL125B.gif';
                        break;
                    default :
                        $file = 'LEVEL1B.gif';
                }
                break;
            default :
                // 該当がない場合、エラー処理として軽症画像を返す。
                $file = 'LEVEL1B.gif';
        }
        $result = APP_ROOT . '/images/' . $file;
        return $result;
    }

    /**
     * Get Shochi Taio Comment
     * @param integer $answer_id : 回答ID
     * @return array Comments
     */
    public function getShochiTaioComment($answer_id)
    {
        $isChild = ($this->mode == self::CHILD);
        $sev_info = Factory::load('DB/DAO_severitys')->getInstance()->fetch_LEFT_JOIN_answers_BY_answer_id($answer_id);
        $zenshin_category_id = $sev_info['zenshin_category_id'];
        $shojo_category_id   = $sev_info['shojo_category_id'];
        $step2_answers   = explode(',', $sev_info['step2_answers']);
        $step31_answers  = explode(',', $sev_info['step31_answers']);
        $step212_answers = explode(',', $sev_info['step212_answers']);
        $fever_answers   = explode(',', $sev_info['fever_answers']);
        $primary_shojo   = $sev_info['primaryShojo'];
        $nowtemp         = $sev_info['nowtemp'];

        $this->prepare($zenshin_category_id, $shojo_category_id);
        $this->execute($step2_answers, $step31_answers, $step212_answers, $fever_answers);

        $result = array(
            'diagnosis_text'        => $this->getDiagnosis(),
            'fever_diagnosis'       => $this->getfeverDiagnosis(),
            'zenshin_diagnosis_arr' => $this->getZenshinDiagnosis($step2_answers),
            'shojo_diagnosis_arr'   => $this->getShojoDiagnosis($step31_answers),
            'gaisho_diagnosis_arr'  => $isChild ? $this->getGaishoDiagnosis($step212_answers) : array(),
            'gaisho_detail_diagnosis_arr' => $isChild ? $this->getGaishoDetailDiagnosis($step212_answers) : array(),
            'goin_diagnosis_text'         => $isChild ? $this->getGoinDiagnosis($step212_answers) : array(),
        );
        return $result;
    }

    /**
     * Get Background Color for List
     * @param int $severity_level 重症度
     * @param bool $has_carte 外来カルテ記録があるか否か
     * @return string Color Code
     */
    public static function getBackgroundColor($severity_level, $has_carte)
    {
        $color = '';
        switch (true) {
            case $has_carte:
                $color = '#DDDDDD';
                break;

            case Severity::isSuperHigh($severity_level):
                $color = '#FF7777';
                break;

            case Severity::isHigh($severity_level):
                $color = '#F4B3C2';
                break;

            case Severity::isMiddle($severity_level):
                //if ((int)$risk >= 5) {
                //    $color = '#F3BF88';
                //} else {
                    $color = '#F8E58C';
                //}
                break;

            case Severity::isLow($severity_level):
                $color = '#EEFFDD';
                break;

            default:
                $color = '';
        }
        return $color;
    }

}


