<?php
/* vim:se et ts=4 sw=4 sts=4 fdm=marker: */
/**
 * Create Xml for Process
 * 
 * LICENSE: 
 * 
 * @copyright 2011 Topazos, Inc.
 * @license http://www.topazos.co.jp/license/3_0.txt PHP License 3.0
 * @version CVS: $Id: $
 * @link http://www.topazos.co.jp/package/PackageName
 * @since File available since Release 1.0.0
 */

require_once dirname(__FILE__) . '/XmlCommon.php';

/**
 * Sample Code
 *
 * $obj = new XmlProcess('Shift_JIS');
 * // Sample Header
 * $header = $obj->parseHeader(
 *     '内科',
 *     '石太郎',
 *     '2011/10/08',
 *     '2011/10/08',
 *     'サンプルデータ',
 *     '9999998'
 * );
 * // Sample INPUTDATA
 * $input_a = array('37.6');
 * $input_b = array('130', '78');
 * $input_c = array('175');
 * $input_d = array('68');
 * $input_e = array('異常なし');
 * // Sample Row
 * $row_a = $obj->parseRow($input_a, '76');
 * $row_b = $obj->parseRow($input_b, '77');
 * $row_c = $obj->parseRow($input_c, '79');
 * $row_d = $obj->parseRow($input_d, '79');
 * $row_e = $obj->parseRow($input_e, '138');
 * // Set Data
 * $obj->setHeader($header);
 * $obj->setRow($row_a);
 * $obj->setRow($row_b);
 * $obj->setRow($row_c);
 * $obj->setRow($row_d);
 * $obj->setRow($row_e);
 * // Execute
 * $xml = $obj->execute();
 * $obj->save($xml); // 第2引数は'dev'or'prod'を指定(デフォルト: dev)
 * 
 */


/**
 * Xml Process Class
 */
class XmlProcess extends XmlCommon
{

    /**
     * Variables
     */
    private $_header = array();  // ヘッダー情報(基本情報)
    private $_row    = array();  // 項目情報
    private $_patient_code = ''; // 患者コード

    /**
     * Constructor
     */
    public function __construct($enc = 'UTF-8')
    {
        parent::__construct($enc);
    }

    /**
     * Execute
     */
    public function execute()
    {
        $data = array();
        // Header...
        $data['Header'] = $this->_getHeader();
        // Row...
        foreach ($this->_row as $val) {
            $data['Content'][0][]['Row'] = $this->_getRow($val);
        }
        $data = $this->_removeSjisGaiji($data);
        $el = $this->_xmlElement('MCSSData', $data, array('ver' => 'Ver.02.00 ' . date('Y-m-d')));
        $this->x->appendChild($el);
        $result = $this->x->saveXML();
        if ($this->_enc != 'UTF-8') {
            $result = $this->_outEncode($result, $this->_enc);
        }
        return $result;
    }

    // {{{ Parser...
    public function parseHeader($depart, $doctor, $consult_date, $exam_date, $comment, $patient_code)
    {
        $this->_patient_code = $patient_code;
        $header = array(
            //'ContentType' => $type,
            //'FileVersion' => $ver,
            'DepartmentName' => $depart,
            'DoctorName' => $doctor,
            'ConsultationDate' => $consult_date,
            'ExaminationDate' => $exam_date,
            'Comment' => $comment,
            'PatientCode' => $patient_code,
            //'InquiryInpDataFileID' => $inquiry,
        );
        return $header;
    }

    public function parseRow(array $input, $master)
    {
        $units = array(
                    'INPUTDATA' => $input,
                    'MasterID' => $master,
                );
        return $units;
    }
    // }}}

    // {{{ Setter...
    /**
     * Set Header
     */
    public function setHeader($data)
    {
        $this->_header = $data;
    }

    /**
     * Set Row
     */
    public function setRow($data)
    {
        $this->_row[] = $data;
    }
    // }}}

    // {{{ Node Tools...
    /**
     * Get Header
     */
    private function _getHeader()
    {
        $header = $this->_header;
        $data[0] = array(
                    'ContentType'      => array('FactInputData'), // FactInputData 固定
                    //'FileVersion'      => array($header['FileVersion']),
                    'FileVersion'      => array('01.00'), // 01.00 固定?要確認
                    'DepartmentName'   => array($header['DepartmentName']),
                    'DoctorName'       => array($header['DoctorName']),
                    'ConsultationDate' => array($header['ConsultationDate']),
                    'ExaminationDate'  => array($header['ExaminationDate']),
                    'Comment'          => array($header['Comment']),
                    'PatientCode'      => array($header['PatientCode']),
                    'InquiryInpDataFileID'  => array('0'), // 0 固定
                    );
        return  $data;
    }

    /**
     * Get Row
     */
    private function _getRow($row)
    {
        $seq_no = 1; // Auto Count Up...
        foreach ($row['INPUTDATA'] as $val) {
            $data[0][]['INPUTDATA'] =  $this->_getINPUTDATA($val, $seq_no++);
        }
        $data[1] = array(
                        'RowCount' => $seq_no - 1,
                        'MasterID' => $row['MasterID'],
                    );
        return $data;
    }

    /**
     * Get INPUTDATA
     */
    private function _getINPUTDATA($input, $seq_no)
    {
        $data[0] = $input;
        $data[1] = array('SeqNo'  => $seq_no);
        return $data;
    }
    // }}}

    // {{{ Save Tools...
    /**
     * 指定ディレクトリの確認作成
     * 
     * ディレクトリが存在しない場合は作成を試みます。
     * 
     * @param string  $inDirPath : ディレクトリパス
     * @param integer $inPerm    : パーミッション
     * 
     * @return mixed (成功)ディレクトリパス or (失敗)FALSE
     * @access public
     */
    function prepareDir($inDirPath, $inPerm)
    {
        $theResult = TRUE;
        if (! is_dir($inDirPath)) {
            umask(0);
            $theResult = @mkdir($inDirPath, $inPerm);
        } else {
            clearstatcache();
            if (is_writable($inDirPath)) {
                $thePerm = fileperms($inDirPath);
                if ($thePerm != $inPerm) {
                    $theResult = @chmod($inDirPath, $inPerm);
                }
            } else {
                $theResult = FALSE;
            }
        }
        if ($theResult) {
            $theResult = $inDirPath;
        }
        return $theResult;
    }

    /**
     * Auto Prepare Directory Recurcive
     * 
     * Try to Create Dir and Modify Permission
     * 
     * @param string  $inDirPath    : Main Directory Path
     * @param integer $inPerm       : Permission to Create Dir
     * @param string  $inParentPath : Parent Path
     * @param integer $inChildPath  : Child Path
     * 
     * @return mixed Main Directory Path or FALSE
     * @see prepareDir
     * @access public
     */
    function autoPrepareDir($inDirPath, $inPerm, $inParentPath = FALSE, $inChildPath = FALSE)
    {
        $theResult = FALSE;
        $theParent = ($inParentPath !== FALSE ? $inParentPath : $inDirPath);
        $theChild  = ($inChildPath !== FALSE ? $inChildPath : $theParent);
        if (! is_dir($theParent)) {
            $theResult = $this->autoPrepareDir($inDirPath, $inPerm, dirname($theParent), $theParent);
            if ($inDirPath != $theResult) {
                $theResult = $this->autoPrepareDir($inDirPath, $inPerm, $theParent, $theChild);
            }
        } else {
            $theResult = $this->prepareDir($theChild, $inPerm);
        }
        return $theResult;
    }

    private function _parseSaveTime($date)
    {
        sscanf($date, "%04d-%02d-%02d %02d:%02d:%02d", $y,$m,$d,$h,$i,$s);
        $result = sprintf("%04d%02d%02d%02d%02d%02d", $y,$m,$d,$h,$i,$s);
        return $result;
    }

    public function getPatientDir($patient_code)
    {
        $i = 0;
        $dir = array();
        $len = strlen($patient_code);
        $max = 3;
        do {
            $dir[] = substr($patient_code, $i, $max);
            $i += $max;
        } while ($i < $len);
        $result = implode('/', $dir);
        return $result;
    }

    public function save($data, $env = 'dev')
    {
        $base = '/var/www/' . $env . '/xml/process';
        $filename = $this->getFileNameWithPatientDir();
        $file = basename($filename);
        $pdir = $base . '/' . dirname($filename);
        $this->autoPrepareDir($pdir, 0777);
        $path = $pdir . '/' . $file;
        $i = 1;
        if (($fp = fopen($path, 'w')) !== FALSE) {
            if (flock($fp, LOCK_EX)) {
                fwrite($fp, $data);
                flock($fp, LOCK_UN);
            }
            fclose($fp);
            chmod($path, 0666);
        }
    }

    public function getFileNameWithPatientDir()
    {
        $patient_code = $this->_patient_code;
        $outdate = date('Y-m-d H:i:s');
        $file = $this->_parseSaveTime($outdate) . '.xml'; // 記録日時
        $pdir = $this->getPatientDir($patient_code);
        $filename = $pdir . '/' . $file;
        return $filename;
    }
    // }}}

}

?>
