<?php
/**
 * データアクセスオブジェクト
 */

require_once dirname(__FILE__).'/PDO_MySQL.php';

/**
 *
 * @see IMD_WDK事前問診_機能仕様書v1.0.doc
 *
 */
class DAO_monshin_adult extends PDO_MySQL
{
    public function __construct()
    {
        parent::__construct(DB_HOST, DB_NAME, DB_USER, DB_PASS, DB_ENCODING);
    }

    /**
     * step2: 全身状態用性年齢区分を返却する
     *
     * @param int $days 生存日数
     */
    public function step2_getAgeDiv($days)
    {
        $sql = "
            SELECT CategoryID,CategoryName FROM INT_NCAgeCategoryTable 
                WHERE MaxDay >= :max_day AND MinDay <= :min_day AND ValidFlag = 0
        ";
        $params = array(
            ':max_day' => $days,
            ':min_day' => $days,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step2: 質問を返却する
     *
     * @param int $category_id カテゴリID
     */
    public function step2_getQuestions($category_id)
    {
        $sql = "
            SELECT ClassID,GQContents FROM INT_NCGQConditionRelationTable GQCRT, INT_NCGQDetailTable GQDT 
                WHERE GQCRT.CategoryID = :category_id AND GQCRT.GQContentsID = GQDT.GQContentsID 
                    ORDER BY GQCRT.ClassID
        ";
        $params = array(
            ':category_id' => $category_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step2: 回答を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $class_id クラスID
     */
    public function step2_getAnswers($category_id, $class_id)
    {
        $sql = "
            SELECT GAnswerID,GACRT.GAContentsID,GAContents FROM INT_NCGAConditionRelationTable GACRT, INT_NCGADetailTable GADT 
                WHERE GACRT.CategoryID = :category_id AND GACRT.ClassID = :class_id AND GACRT.GAContentsID = GADT.GAContentsID 
                    ORDER BY GACRT.GAnswerID
        ";
        $params = array(
            ':category_id' => $category_id,
            ':class_id' => $class_id,
        );
        return $this->fetchAll($sql, $params);
    }



    /**
     * step3: カテゴリIDを返却する
     *
     * @param int $days 生存日数
     * @param int $gender 性別 (1:男 / 2:女)
     */
    public function step3_getCategoryId($days, $gender)
    {
        // 男4以下、女5以上
        $where = ($gender == 1) ? 'CategoryID <= 4' : 'CategoryID >= 5';

        $sql = "
            SELECT CategoryID,CategoryName FROM INT_AgeCategoryTable 
                WHERE MaxDay >= :max_day AND MinDay <= :min_day AND {$where} AND ValidFlag = 0
        ";
        $params = array(
            ':max_day' => $days,
            ':min_day' => $days,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step3: 症状項目を返却する
     *
     * @param int $category_id カテゴリID
     */
    public function step3_getDiseaseItems($category_id)
    {
        /*$sql = "
            SELECT DiseaseID,DiseaseName FROM INT_DiseaseNameTable 
                WHERE CategoryID = :category_id AND ValidFlag = 0 ORDER BY DiseaseID
                limit 0, 24
        ";*/
        $sql = "
            SELECT DiseaseID,DiseaseName,OrgFlag FROM INT_DiseaseNameTable 
                WHERE CategoryID = :category_id AND ValidFlag = 1 ORDER BY orderby
        ";
        /*$sql = "
            SELECT DiseaseID,DiseaseName,OrgFlag FROM INT_DiseaseNameTable name
                LEFT JOIN DiseaseCategoryTable cate ON name.DisCatID = cate.DisCatID
                    WHERE CategoryID = :category_id ORDER BY cate.DisCatID, DiseaseID limit 0, 24
        ";*/
        $params = array(
            ':category_id' => $category_id,
        );

        return $this->fetchAll($sql, $params);
    }

    /**
     * step31: 症状を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $disease_id 症状ID
     */
    public function step31_getDiseaseItem($category_id, $disease_id)
    {
        /*$sql = "
            SELECT DiseaseID,DiseaseName FROM INT_DiseaseNameTable 
                WHERE CategoryID = :category_id AND DiseaseID = :disease_id AND ValidFlag = 0
        ";*/
        $sql = "
            SELECT DiseaseID,DiseaseName,OrgFlag  FROM INT_DiseaseNameTable 
                WHERE CategoryID = :category_id AND DiseaseID = :disease_id
        ";
        $params = array(
            ':category_id' => $category_id,
            ':disease_id' => $disease_id,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step31: 質問を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $disease_id 症状ID
     */
    public function step31_getQestions($category_id, $disease_id)
    {
        $sql = "
            SELECT DQuestionID,DQContents,when_flag FROM INT_DQExaminationTable 
                WHERE CategoryID = :category_id AND DiseaseID = :disease_id ORDER BY DQuestionID
                limit 0, 3
        ";
        $params = array(
            ':category_id' => $category_id,
            ':disease_id' => $disease_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step31: 回答を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $question_id 質問ID
     */
    public function step31_getAnswers($category_id, $question_id)
    {
        $sql = "
            SELECT DAnswerID,CategoryID,DAContents,when_id FROM INT_DAExaminationTable 
                WHERE CategoryID = :category_id AND DQuestionID = :question_id ORDER BY DAnswerID
                limit 0, 3
        ";
        $params = array(
            ':category_id' => $category_id,
            ':question_id' => $question_id,
        );
        return $this->fetchAll($sql, $params);
    }



    /**
     * step6: 全身状態の重症度判定結果を抽出
     *
     * @param int $consciousness_id 
     * @param int $breath_id 
     * @param int $dialog_id 
     * @param int $activity_id 
     * @param int $dehydration_id 
     * @param int $category_id 
     *
     * @see いただいたドキュメント/重症度／緊急度(重症化リスク)/16歳以上のとき関連クエリ.txt
     */
    public function step6_getZenshinSeverity(
        $consciousness_id, $breath_id, $dialog_id, $activity_id, $dehydration_id, $category_id)
    {
        /*
        SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM INT_NCGQAResultTable CRsT
        LEFT OUTER JOIN INT_ConsultNameTable CNT ON CNT.DiagnosisID = CRsT.DiagnosisID
        WHERE ConsciousnessLNID = 全身状態Q1の回答
        AND   BreathLNID        = 全身状態Q2の回答
        AND   DialogLNID        = 全身状態Q3の回答
        AND   ActivityLNID      = 全身状態Q4の回答
        AND   DehydrationLNID   = 全身状態Q5の回答
        AND   CategoryID        = 16歳以上の場合の性年齢区分(INT_NCAgeCategoryTableより＊INT_AgeCategoryTableではない→全身状態のときはNCxx、小児と同一)
        */
        $sql = "
            SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM INT_NCGQAResultTable CRsT
                LEFT OUTER JOIN INT_ConsultNameTable CNT ON CNT.DiagnosisID = CRsT.DiagnosisID
                    WHERE ConsciousnessLNID = :consciousness_id
                        AND BreathLNID      = :breath_id
                        AND DialogLNID      = :dialog_id
                        AND ActivityLNID    = :activity_id
                        AND DehydrationLNID = :dehydration_id
                        AND CategoryID      = :category_id
        ";

        $params = array(
            ':consciousness_id' => $consciousness_id,
            ':breath_id' => $breath_id,
            ':dialog_id' => $dialog_id,
            ':activity_id' => $activity_id,
            ':dehydration_id' => $dehydration_id,
            ':category_id' => $category_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 症状選択1の重症度判定結果を抽出
     *
     * @param array $answer_ids 回答ID
     * @param int $category_id カテゴリID
     *
     * @see いただいたドキュメント/重症度／緊急度(重症化リスク)/16歳以上のとき関連クエリ.txt
     */
    public function step6_getShojoSeverity(Array $answer_ids, $category_id)
    {
        $params = array();

        $cond = ' DAnswerID = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' OR DAnswerID = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' OR DAnswerID = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM INT_ConsultNameTable CNT, INT_ConsultResultTable CRsT 
                WHERE CRsT.TDRelationID = (
                    SELECT RELATION.TDRelationID FROM (
                        SELECT M1.TDRelationID,COUNT(*) AS COUNTRECORD FROM INT_ConsultRelationTable M1 
                            WHERE {$cond} GROUP BY M1.TDRelationID
                    ) AS RELATION WHERE COUNTRECORD = 3
                ) AND CRsT.CategoryID = :category_id AND CNT.DiagnosisID = CRsT.DiagnosisID ORDER BY 1 DESC LIMIT 0, 1
        ";

        $params[':category_id'] = $category_id;

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 外傷選択3の重症度判定結果を抽出
     *
     * @param array $answer_ids 回答ID
     */
    public function step6_getGaishoSeverity(Array $answer_ids)
    {
        $params = array();

        $cond = ' Result1 = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' AND Result2 = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' AND Result3 = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM INI_IQAResultTable CRsT 
                LEFT OUTER JOIN INT_ConsultNameTable CNT ON CNT.DiagnosisID=CRsT.DiagnosisID 
                    WHERE {$cond}
        ";

        return $this->fetch($sql, $params);
    }

    //
    // 大人用では重症度判定に発熱を利用しない。
    //
    /**
     * step6: 発熱の重症度判定結果を抽出
     *
     * @param array $answer_ids 回答ID
     * @param int $category_id カテゴリID
     */
    /*public function step6_getFeverSeverity(Array $answer_ids, $category_id)
    {
        $params = array();

        $cond = ' DAnswerID = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' OR DAnswerID = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' OR DAnswerID = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT * FROM ConsultNameTable CNT, ConsultResultTable CRsT 
                WHERE CRsT.TDRelationID = (
                    SELECT RELATION.TDRelationID FROM (
                        SELECT M1.TDRelationID,COUNT(*) AS COUNTRECORD FROM ConsultRelationTable M1 
                            WHERE {$cond} GROUP BY M1.TDRelationID
                    ) AS RELATION WHERE COUNTRECORD=3
                ) AND CRsT.CategoryID = :category_id AND CNT.DiagnosisID=CRsT.DiagnosisID ORDER BY 1 DESC LIMIT 0, 1
        ";

        $params[':category_id'] = $category_id;

        return $this->fetch($sql, $params);
    }*/

    /**
     * step6: 全身状態の処置対応を抽出
     *
     * @param int $category_id 
     * @param int $class_id 
     * @param int $answer_id 
     */
    public function step6_getZenshinDiagnosis($category_id, $class_id, $answer_id)
    {
        $sql = "
            SELECT GAContents,GAParameter,TreatmentContents FROM INT_NCGAConditionRelationTable GACRT, NCGADetailTable GADT 
                WHERE GACRT.CategoryID = :category_id AND GACRT.ClassID = :class_id AND 
                    GACRT.GAnswerID = :answer_id AND GACRT.GAContentsID = GADT.GAContentsID
        ";
        $params = array(
            ':category_id' => $category_id,
            ':class_id' => $class_id,
            ':answer_id' => $answer_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 症状選択の処置対応を抽出
     *
     * @param array $answer_ids 回答ID
     * @param int $category_id カテゴリID
     */
    public function step6_getShojoDiagnosis(Array $answer_ids, $category_id)
    {
        $params = array();

        $cond = ' DAnswerID = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' OR DAnswerID = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' OR DAnswerID = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT CNT.DiagnosisID,CRsT.TreatmentName FROM INT_ConsultNameTable CNT, INT_ConsultResultTable CRsT 
                WHERE CRsT.TDRelationID = (
                    SELECT RELATION.TDRelationID FROM (
                        SELECT M1.TDRelationID,COUNT(*) AS COUNTRECORD FROM INT_ConsultRelationTable M1 
                            WHERE {$cond} GROUP BY M1.TDRelationID
                    ) AS RELATION WHERE COUNTRECORD = 3
                ) AND CRsT.CategoryID = :category_id AND CNT.DiagnosisID = CRsT.DiagnosisID ORDER BY 1 DESC LIMIT 0, 1
        ";

        $params[':category_id'] = $category_id;

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 外傷選択の処置対応を抽出①
     *
     * @param int $injury_id 
     */
    public function step6_getGaishoDiagnosis($injury_id)
    {
        $sql = "
            SELECT InjuryID,InjuryCommonTreat FROM INI_InjuryNameTable WHERE InjuryID = :injury_id
        ";
        $params = array(
            ':injury_id' => $injury_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 外傷選択の処置対応を抽出②
     *
     * @param int $res1_id 
     * @param int $res2_id
     * @param int $res3_id
     */
    public function step6_getGaishoDetailDiagnosis($res1_id, $res2_id, $res3_id)
    {
        $sql = "
            SELECT CRsT.InjuryDetailTreat FROM INI_IQAResultTable CRsT 
                LEFT OUTER JOIN ConsultNameTable CNT ON CNT.DiagnosisID = CRsT.DiagnosisID 
                    WHERE Result1 = :res1_id AND Result2 = :res2_id AND Result3 = :res3_id
        ";
        $params = array(
            ':res1_id' => $res1_id,
            ':res2_id' => $res2_id,
            ':res3_id' => $res3_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 誤飲選択の処置対応を抽出
     *
     * @param int $object_id 
     */
    public function step6_getGoinDiagnosis($object_id)
    {
        $sql = "
            SELECT SwallowTreatmentClass,SwallowTreatmentContents FROM INI_SwallowTreatmentTable 
                WHERE ValidFlag = 0 And SwallowObjectID = :object_id ORDER BY SwallowTreatmentClass
        ";
        $params = array(
            ':object_id' => $object_id,
        );

        return $this->fetch($sql, $params);
    }

} //-- End of class


