<?php
/**
 * データアクセスオブジェクト
 */

require_once dirname(__FILE__).'/PDO_MySQL.php';

/**
 *
 * @see IMD_WDK事前問診_機能仕様書v1.0.doc
 *
 */
class DAO_monshin_child extends PDO_MySQL
{
    public function __construct()
    {
        parent::__construct(DB_HOST, DB_NAME, DB_USER, DB_PASS, DB_ENCODING);
    }

    /**
     * step2: 全身状態用性年齢区分を返却する
     *
     * @param int $days 生存日数
     */
    public function step2_getAgeDiv($days)
    {
        $sql = "
            SELECT CategoryID,CategoryName FROM NCAgeCategoryTable 
                WHERE MaxDay >= :max_day AND MinDay <= :min_day AND ValidFlag = 0
        ";
        $params = array(
            ':max_day' => $days,
            ':min_day' => $days,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step2: 質問を返却する
     *
     * @param int $category_id カテゴリID
     */
    public function step2_getQuestions($category_id)
    {
        $sql = "
            SELECT ClassID,GQContents FROM NCGQConditionRelationTable GQCRT, NCGQDetailTable GQDT 
                WHERE GQCRT.CategoryID = :category_id AND GQCRT.GQContentsID = GQDT.GQContentsID 
                    ORDER BY GQCRT.ClassID
        ";
        $params = array(
            ':category_id' => $category_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step2: 回答を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $class_id クラスID
     */
    public function step2_getAnswers($category_id, $class_id)
    {
        $sql = "
            SELECT GAnswerID,GACRT.GAContentsID,GAContents FROM NCGAConditionRelationTable GACRT, NCGADetailTable GADT 
                WHERE GACRT.CategoryID = :category_id AND GACRT.ClassID = :class_id AND GACRT.GAContentsID = GADT.GAContentsID 
                    ORDER BY GACRT.GAnswerID
        ";
        $params = array(
            ':category_id' => $category_id,
            ':class_id' => $class_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step211: 質問を返却する
     */
    public function step211_getQuestions()
    {
        $sql = "
            SELECT InjuryID,InjuryName,InjFileName FROM InjuryNameTable WHERE ValidFlag = 0 ORDER BY InjuryID
        ";
        $params = array();
        return $this->fetchAll($sql, $params);
    }



    /**
     * step212: 外傷を返却する
     *
     * @param int $injury_id 外傷ID
     */
    public function step212_getInjury($injury_id)
    {
        $sql = "
            SELECT InjuryID,InjuryName,InjFileName FROM InjuryNameTable WHERE ValidFlag = 0 AND InjuryID = :injury_id
        ";
        $params = array(
            ':injury_id' => $injury_id,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step212: 質問を返却する
     *
     * @param int $injury_id 外傷ID
     */
    public function step212_getQuestions($injury_id)
    {
        $sql = "
            SELECT IQuestionID,IQContents FROM IQExaminationTable WHERE InjuryID = :injury_id ORDER BY IQuestionID
            limit 0, 3
        ";
        $params = array(
            ':injury_id' => $injury_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step212: 回答を返却する
     *
     * @param int $question_id 質問ID
     */
    public function step212_getAnswers($question_id)
    {
        $sql = "
            SELECT IAnswerID,IAContents FROM IAExaminationTable WHERE IQuestionID = :question_id ORDER BY IAnswerID
            limit 0, 3
        ";
        $params = array(
            ':question_id' => $question_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step22: 選択肢を返却する
     * 
     * @param array $ids 選択肢ID
     */
    public function step22_getItems(Array $ids)
    {
        $ids = join(',', $ids);
        $sql = "
            SELECT SwallowObjectID,SwallowObjectName FROM SwallowExaminationTable WHERE SwallowClass IN ({$ids}) 
                ORDER BY SwallowObjectID
        ";
        $params = array();
        return $this->fetchAll($sql, $params);
    }



    /**
     * step3: カテゴリIDを返却する
     *
     * @param int $days 生存日数
     * @param int $gender 性別 (1:男 / 2:女)
     */
    public function step3_getCategoryId($days, $gender)
    {
        // 男10以下、女11以上
        $where = ($gender == 1) ? 'CategoryID <= 10' : 'CategoryID >= 11';

        $sql = "
            SELECT CategoryID,CategoryName FROM AgeCategoryTable 
                WHERE MaxDay >= :max_day AND MinDay <= :min_day AND {$where} AND ValidFlag = 0
        ";
        $params = array(
            ':max_day' => $days,
            ':min_day' => $days,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step3: 症状項目を返却する
     *
     * @param int $category_id カテゴリID
     */
    public function step3_getDiseaseItems($category_id)
    {
        $sql = "
            SELECT * FROM DiseaseNameTable name
                LEFT JOIN DiseaseCategoryTable cate ON name.DisCatID = cate.DisCatID
                    WHERE CategoryID = :category_id AND ValidFlag = 0 ORDER BY name.DisCatID, DiseaseID
        ";
        $params = array(
            ':category_id' => $category_id,
        );
        return $this->fetchAll($sql, $params);
    }


    /**
     * step31: 症状を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $disease_id 症状ID
     */
    public function step31_getDiseaseItem($category_id, $disease_id)
    {
        if ($disease_id == FEVER_ID) {
            // 「発熱」の場合： ValidFlag = 0 不要
            $sql = "
                SELECT DiseaseID,DiseaseName FROM DiseaseNameTable 
                    WHERE CategoryID = :category_id AND DiseaseID = :disease_id
            ";
        } else {
            $sql = "
                SELECT DiseaseID,DiseaseName FROM DiseaseNameTable 
                    WHERE CategoryID = :category_id AND DiseaseID = :disease_id AND ValidFlag = 0
            ";
        }
        $params = array(
            ':category_id' => $category_id,
            ':disease_id' => $disease_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step31: 質問を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $disease_id 症状ID
     */
    public function step31_getQestions($category_id, $disease_id)
    {
        $sql = "
            SELECT DQuestionID,DQContents,when_flag FROM DQExaminationTable 
                WHERE CategoryID = :category_id AND DiseaseID = :disease_id ORDER BY DQuestionID
                limit 0, 3
        ";
        $params = array(
            ':category_id' => $category_id,
            ':disease_id' => $disease_id,
        );
        return $this->fetchAll($sql, $params);
    }

    /**
     * step31: 回答を返却する
     *
     * @param int $category_id カテゴリID
     * @param int $question_id 質問ID
     */
    public function step31_getAnswers($category_id, $question_id)
    {
        $sql = "
            SELECT DAnswerID,CategoryID,DAContents,when_id FROM DAExaminationTable 
                WHERE CategoryID = :category_id AND DQuestionID = :question_id ORDER BY DAnswerID
                limit 0, 3
        ";
        $params = array(
            ':category_id' => $category_id,
            ':question_id' => $question_id,
        );
        return $this->fetchAll($sql, $params);
    }



    /**
     * step6: 全身状態の重症度判定結果を抽出
     *
     * @param int $injury_id 
     * @param int $consciousness_id 
     * @param int $breath_id 
     * @param int $activity_id 
     * @param int $dehydration_id 
     * @param int $category_id 
     */
    public function step6_getZenshinSeverity(
        $injury_id, $consciousness_id, $breath_id, $activity_id, $dehydration_id, $category_id)
    {
        $sql = "
            SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM NCGQAResultTable CRsT 
                LEFT OUTER JOIN ConsultNameTable CNT ON CNT.DiagnosisID=CRsT.DiagnosisID 
                    WHERE InjuryID = :injury_id AND ConsciousnessID = :consciousness_id AND 
                        BreathID = :breath_id AND ActivityID = :activity_id AND 
                            DehydrationID = :dehydration_id AND CategoryID = :category_id
        ";
        $params = array(
            ':injury_id' => $injury_id,
            ':consciousness_id' => $consciousness_id,
            ':breath_id' => $breath_id,
            ':activity_id' => $activity_id,
            ':dehydration_id' => $dehydration_id,
            ':category_id' => $category_id,
        );
        return $this->fetch($sql, $params);
    }

    /**
     * step6: 症状選択1の重症度判定結果を抽出
     *
     * @param array $answer_ids 回答ID
     * @param int $category_id カテゴリID
     */
    public function step6_getShojoSeverity(Array $answer_ids, $category_id)
    {
        $params = array();

        $cond = ' DAnswerID = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' OR DAnswerID = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' OR DAnswerID = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM ConsultNameTable CNT, ConsultResultTable CRsT 
                WHERE CRsT.TDRelationID = (
                    SELECT RELATION.TDRelationID FROM (
                        SELECT M1.TDRelationID,COUNT(*) AS COUNTRECORD FROM ConsultRelationTable M1 
                            WHERE {$cond} GROUP BY M1.TDRelationID
                    ) AS RELATION WHERE COUNTRECORD = 3
                ) AND CRsT.CategoryID = :category_id AND CNT.DiagnosisID = CRsT.DiagnosisID ORDER BY 1 DESC LIMIT 0, 1
        ";

        $params[':category_id'] = $category_id;

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 外傷選択3の重症度判定結果を抽出
     *
     * @param array $answer_ids 回答ID
     */
    public function step6_getGaishoSeverity(Array $answer_ids)
    {
        $params = array();

        $cond = ' Result1 = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' AND Result2 = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' AND Result3 = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT CNT.DiagnosisID,CNT.DiagnosisContents FROM IQAResultTable CRsT 
                LEFT OUTER JOIN ConsultNameTable CNT ON CNT.DiagnosisID=CRsT.DiagnosisID 
                    WHERE {$cond}
        ";

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 発熱の重症度判定結果を抽出
     *
     * @param array $answer_ids 回答ID
     * @param int $category_id カテゴリID
     */
    public function step6_getFeverSeverity(Array $answer_ids, $category_id)
    {
        $params = array();

        $cond = ' DAnswerID = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' OR DAnswerID = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' OR DAnswerID = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT * FROM ConsultNameTable CNT, ConsultResultTable CRsT 
                WHERE CRsT.TDRelationID = (
                    SELECT RELATION.TDRelationID FROM (
                        SELECT M1.TDRelationID,COUNT(*) AS COUNTRECORD FROM ConsultRelationTable M1 
                            WHERE {$cond} GROUP BY M1.TDRelationID
                    ) AS RELATION WHERE COUNTRECORD=3
                ) AND CRsT.CategoryID = :category_id AND CNT.DiagnosisID=CRsT.DiagnosisID ORDER BY 1 DESC LIMIT 0, 1
        ";

        $params[':category_id'] = $category_id;

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 全身状態の処置対応を抽出
     *
     * @param int $category_id 
     * @param int $class_id 
     * @param int $answer_id 
     */
    public function step6_getZenshinDiagnosis($category_id, $class_id, $answer_id)
    {
        $sql = "
            SELECT GAContents,GAParameter,TreatmentContents FROM NCGAConditionRelationTable GACRT, NCGADetailTable GADT 
                WHERE GACRT.CategoryID = :category_id AND GACRT.ClassID = :class_id AND 
                    GACRT.GAnswerID = :answer_id AND GACRT.GAContentsID = GADT.GAContentsID
        ";
        $params = array(
            ':category_id' => $category_id,
            ':class_id' => $class_id,
            ':answer_id' => $answer_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 症状選択の処置対応を抽出
     *
     * @param array $answer_ids 回答ID
     * @param int $category_id カテゴリID
     */
    public function step6_getShojoDiagnosis(Array $answer_ids, $category_id)
    {
        $params = array();

        $cond = ' DAnswerID = :ans1_id ';
        $params[':ans1_id'] = $answer_ids[0];

        if (isset($answer_ids[1])) {
            $cond .= ' OR DAnswerID = :ans2_id ';
            $params[':ans2_id'] = $answer_ids[1];
        }

        if (isset($answer_ids[2])) {
            $cond .= ' OR DAnswerID = :ans3_id ';
            $params[':ans3_id'] = $answer_ids[2];
        }

        $sql = "
            SELECT CNT.DiagnosisID,CRsT.TreatmentName FROM ConsultNameTable CNT, ConsultResultTable CRsT 
                WHERE CRsT.TDRelationID = (
                    SELECT RELATION.TDRelationID FROM (
                        SELECT M1.TDRelationID,COUNT(*) AS COUNTRECORD FROM ConsultRelationTable M1 
                            WHERE {$cond} GROUP BY M1.TDRelationID
                    ) AS RELATION WHERE COUNTRECORD = 3
                ) AND CRsT.CategoryID = :category_id AND CNT.DiagnosisID = CRsT.DiagnosisID ORDER BY 1 DESC LIMIT 0, 1
        ";

        $params[':category_id'] = $category_id;

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 外傷選択の処置対応を抽出①
     *
     * @param int $injury_id 
     */
    public function step6_getGaishoDiagnosis($injury_id)
    {
        $sql = "
            SELECT InjuryID,InjuryCommonTreat FROM InjuryNameTable WHERE InjuryID = :injury_id
        ";
        $params = array(
            ':injury_id' => $injury_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 外傷選択の処置対応を抽出②
     *
     * @param int $res1_id 
     * @param int $res2_id
     * @param int $res3_id
     */
    public function step6_getGaishoDetailDiagnosis($res1_id, $res2_id, $res3_id)
    {
        $sql = "
            SELECT CRsT.InjuryDetailTreat FROM IQAResultTable CRsT 
                LEFT OUTER JOIN ConsultNameTable CNT ON CNT.DiagnosisID = CRsT.DiagnosisID 
                    WHERE Result1 = :res1_id AND Result2 = :res2_id AND Result3 = :res3_id
        ";
        $params = array(
            ':res1_id' => $res1_id,
            ':res2_id' => $res2_id,
            ':res3_id' => $res3_id,
        );

        return $this->fetch($sql, $params);
    }

    /**
     * step6: 誤飲選択の処置対応を抽出
     *
     * @param int $object_id 
     */
    public function step6_getGoinDiagnosis($object_id)
    {
        $sql = "
            SELECT SwallowTreatmentClass,SwallowTreatmentContents FROM SwallowTreatmentTable 
                WHERE ValidFlag = 0 And SwallowObjectID = :object_id ORDER BY SwallowTreatmentClass
        ";
        $params = array(
            ':object_id' => $object_id,
        );

        return $this->fetch($sql, $params);
    }

} //-- End of class


