<?php
/**
 * PDO_MySQL_Manipulator Class
 *
 * @package Lambda/lib/DB/PDO
 * @author  rooth
 * @version 0.0.1
 *
 * PHP versions 5
 *
 *<pre>
 *
 * usage:
 *
 * [common] ----------------------------------------------------------------------------------------
 * $this->manipulator = new PDO_MySQL_Manipulator(DB_HOST, DB_NAME, DB_USER, DB_PASS);
 * $this->manipulator->setTable('test_tbl');
 *
 * [select] ----------------------------------------------------------------------------------------
 * $this->manipulator->select('SQL_CALC_FOUND_ROWS *');
 * - or -
 * $this->manipulator->select('a', 'b', 'CCC as count');
 *
 * $this->manipulator->where(array(
 *     '(',
 *     'test_id' => 123, // or 'test_id' = array('>', 123); or array('test_id', '>', 123);
 *     'and',
 *     'test_name' => 'hoge',
 *     'and',
 *     'test_num' => array('BETWEEN|NOT BETWEEN', 1, 9),
 *     'and',
 *     'test_col' => array('IS NULL|IS NOT NULL'),
 *     'and',
 *     'FUNC:FIND_IN_SET' => array(array('column' => 'area_div_id'), array('value' => '1,3')) // FIND_IN_SET(area_div_id, '1,3')
 *     'and',
 *     'FUNC:FIND_IN_SET' => array(array('value' => 1), array('column' => 'genre_div_id'))    // FIND_IN_SET(1, genre_div_id) 
 *     ')',
 * ));
 * - or -
 * $this->manipulator->where(array(
 *     $column => array('IN', $arr),  // or $column => array('NOT IN', $arr),
 *     'and',
 *     'test_id' => 123,
 *     ));
 * - or -
 * // [廃止] $this->manipulator->whereIn($column, $arr);
 *
 * $this->manipulator->groupBy('a', 'b');
 *
 * $this->manipulator->having(); // 未実装
 *
 * $this->manipulator->orderBy(array(
 *     'test_name' => 'DESC',
 *     'test_id'   => 'ASC',
 * ));
 * - or -
 * $this->manipulator->orderBy(array('rand()'));
 *
 * $this->manipulator->limit(0, 10);
 * 
 * $result = $this->manipulator->fetchAll(true/false);
 * - or -
 * $result = $this->manipulator->fetch(true/false);
 * - or -
 * $result = $this->manipulator->fetchColumn(true/false);
 *
 * $total_rows = $this->manipulator->foundRows();
 *
 * [join] ----------------------------------------------------------------------------------------
 * $this->manipulator->setAlias('mc');
 *
 * $this->manipulator->select('
 *     SQL_CALC_FOUND_ROWS *, mc.name as mc_name, mk.name as mk_name, mc.active_flg as mc_active_flg
 * ');
 *
 * $this->manipulator->join('LEFT', 'makers mk ON mc.maker_id = mk.maker_id');
 * - or -
 * $this->manipulator->join('LEFT', 'makers mk');
 * $this->manipulator->using('maker_id');
 *
 * $this->manipulator->where(array(
 *     'mc.type_div_id' => $type_div_id,
 * ));
 *
 * $this->manipulator->orderBy(array(
 *     'mc.name' => 'ASC',
 * ));
 *
 * $result = $this->manipulator->fetchAll(true/false);
 *
 * [insert] ----------------------------------------------------------------------------------------
 * $this->manipulator->columns('id', 'name', 'count');
 *
 * $this->manipulator->values(array(
 *     'id'    => 123,
 *     'name'  => $p['name'], 
 *     'count' => 1, 
 * ));
 *
 * $this->manipulator->onDuplicateKeyUpdate(array(
 *     'name'  => $p['name'], 
 *     'count' => array('=', 'count + 1'), // or 'count' => $count + 1, 
 * ));
 *
 * $result = $this->manipulator->insert(true/false);
 *
 * [update] ----------------------------------------------------------------------------------------
 * $this->manipulator->set(array(
 *     'col_a' => 'hoge',
 *     'col_b' => 'fuga',
 * ));
 *
 * $this->manipulator->where(array(
 *     'test_id' => 123, // or 'test_id' = array('>', 123);
 *     'and',
 *     'test_name' => 'hoge',
 * ));
 * - or -
 * // [廃止] $this->manipulator->whereIn($column, $arr);
 *
 * $result = $this->manipulator->update(true/false);
 *
 * [delete] ----------------------------------------------------------------------------------------
 * $this->manipulator->where(array(
 *     'test_id' => 123, // or 'test_id' = array('>', 123);
 *     'and',
 *     'test_name' => 'hoge',
 * ));
 * - or -
 * // [廃止] $this->manipulator->whereIn($column, $arr);
 *
 * $result = $this->manipulator->delete(true/false);
 * 
 * [raw query] -------------------------------------------------------------------------------------
 * $sql = "select * from $this->table where user_id = $user_id and hoge like '$hoge%' ";
 * $this->manipulator->rawQuery('fetchColmun|fetch|fetchAll|execute', $sql);
 *
 *
 *
 * change log:
 *
 * 2009.09.24 setWhereメソッドを複数回コールできるように拡張。
 * 2009.10.06 joinメソッドを追加。
 * 2009.10.14 where句に BETWEEN|NOT BETWEEN を指定できるように拡張。
 * 2009.10.15 冗長だったメソッド名を修正。 ex) setWhere -> where
 * 2009.10.15 selectメソッドを追加。
 * 2009.10.16 setAliasメソッドを追加。
 * 2009.10.16 usingメソッドを追加。
 * 2010.01.06 setメソッドを複数回コールできるように拡張。
 * 2010.01.25 where句に IS NULL|IS NOT NULL を指定できるように拡張。
 * 2010.02.04 where句に IN|NOT IN を指定できるように拡張。IN + (and|or) のような復号条件を指定可能になった。
 * 2010.02.12 forUpdateメソッドを追加。(SELECT ... FOR UPDATE)
 * 2010.02.12 lock / unlockメソッドを追加。(LOCK TABLES / UNLOCK TABLES)
 * 2010.02.18 updateWithAffectedRowsメソッドを追加。
 * 2010.02.18 deleteWithAffectedRowsメソッドを追加。
 * 2010.02.23 where句において、同名カラムを指定できるように拡張。
 *
 *    例）WHERE reg_date >= start_date AND reg_date <= end_date のような where句 を構築したい場合
 *
 *        // 配列で設定する array(カラム名, 演算子, 値)
 *        $this->manipulator->where(array(
 *            array('reg_date', '>=', $start_date), 
 *            'and', 
 *            array('reg_date', '<=', $end_date), 
 *            'or', 
 *            array('reg_date', 'IS', null), // or array('reg_date', 'IS NOT', null), 
 *        ));
 *
 * 2010.03.09 hasWhereメソッドを追加。
 * 2010.08.16 ifDefineWhereメソッドを追加。usage) $this->manipulator->ifDefineWhere('and');
 * 2010.10.25 join と useing を複数回定義できるように拡張。
 * 2011.03.15 where句に FIND_IN_SET を指定できるように拡張。
 *
 *</pre>
 */

require_once dirname(__FILE__).'/PDO_MySQL.php';

class PDO_MySQL_Manipulator
{
    protected static $pdo;

    protected $table;
    protected $alias;
    protected $columns;
    protected $join;
    protected $using = array();
    protected $where = array();
    //protected $wherein;
    protected $groupby;
    protected $having;
    protected $orderby;
    protected $limit;
    protected $values;
    protected $set;
    protected $params = array();
    protected $query;
    protected $duplicate_key_update;
    protected $for_update;
    private   $_last_query = '';

    const SELECT = 1;
    const INSERT = 2;
    const UPDATE = 3;
    const DELETE = 4;

    public function __construct($host, $db, $user, $pass, $encoding = '')
    {
        if ( ! self::$pdo) self::$pdo = new PDO_MySQL($host, $db, $user, $pass, $encoding);
        $this->initParams();
    }

    protected function initParams()
    {
        $this->_setLastQuery();

        $this->alias   = '';
        $this->columns = '';
        $this->join    = array();
        $this->using   = array();
        $this->where   = '';
        //$this->wherein = '';
        $this->groupby = '';
        $this->having  = '';
        $this->orderby = '';
        $this->limit   = '';
        $this->values  = '';
        $this->set     = '';
        $this->params  = array();
        $this->query   = '';
        $this->duplicate_key_update = '';
        $this->for_update = '';
    }

    /**
     * 対象テーブル
     */
    public function setTable($table)
    {
        $this->table = $table;
    }

    /**
     * エイリアス名 (for join)
     */
    public function setAlias($alias)
    {
        $this->alias = $alias;
    }

    /**
     * select 句
     *  columns() のエイリアス
     */
    public function select()
    {
        $args = func_get_args();
        $this->columns = implode(', ', $args);
    }
    /**
     * カラム (for insert)
     */
    public function columns()
    {
        $args = func_get_args();
        $this->columns = implode(', ', $args);
    }

    /**
     * join 句
     */
    public function join($type, $sql)
    {
        switch (strtoupper(trim($type))) {
            case 'LEFT':
                $this->join[] = " LEFT JOIN {$sql} ";
                break;

            case 'RIGHT':
                $this->join[] = " RIGHT JOIN {$sql} ";
                break;

            case 'INNER':
                $this->join[] = " INNER JOIN {$sql} ";
                break;

            case 'OUTER':
                $this->join[] = " OUTER JOIN {$sql} ";
                break;

            default:
                $this->join[] = '';
        }
    }

    /**
     * using 句
     */
    public function using($column)
    {
        $this->using[] = " USING ({$column}) ";
    }

    /**
     * where 句
     */
    public function where($arr)
    {
        $this->where = ($this->where === '') ? 'WHERE ' : $this->where;
        foreach ($arr as $k => $v) {
            if (preg_match('/^[0-9]+$/', $k)) {
                // eg) (|)|and|or
                if (count($v) === 1 && preg_match('/[()]+|and|or/i', trim($v))) {
                    $this->where .= strtoupper($v).' ';
                    continue;
                }
                // eg) array('column_name', '=', $value) or array('column_name', 'IS/IS NOT', null)
                if (count($v) === 3) {
                    $operator = strtoupper(trim($v[1]));
                    if ($v[2] === null && ($operator === 'IS' || $operator === 'IS NOT')) {
                        $this->where .= " {$v[0]} {$operator} NULL ";
                    } else {
                        $named_key = self::mkNamedKey();
                        $this->where .= " {$v[0]} {$v[1]} {$named_key} ";
                        $this->setParams($named_key, $v[2]);
                    }
                    continue;
                }
            }

            if (is_array($v) && isset($v[0])) {
                $operator = is_array($v[0]) ? $v[0] : strtoupper(trim($v[0]));
                // BETWEEN|NOT BETWEEN
                if ($operator === 'BETWEEN' || $operator === 'NOT BETWEEN') {
                    $min_named_key = self::mkNamedKey();
                    $max_named_key = self::mkNamedKey();
                    $this->where .= " {$k} {$operator} {$min_named_key} AND {$max_named_key} ";
                    $this->setParams($min_named_key, $v[1]);
                    $this->setParams($max_named_key, $v[2]);
                    continue;
                }
                // IS NULL|IS NOT NULL
                if ($operator === 'IS NULL' || $operator === 'IS NOT NULL') {
                    $this->where .= " {$k} {$operator} ";
                    continue;
                }
                // IN|NOT IN
                if ($operator === 'IN' || $operator === 'NOT IN') {
                    $buff = array();
                    foreach ($v[1] as $inv) {
                        $named_key = self::mkNamedKey();
                        $buff[] = $named_key;
                        $this->setParams($named_key, $inv);
                    }
                    $this->where .= " {$k} {$operator} (".implode(', ', $buff).') ';
                    continue;
                }
                // FIND_IN_SET
                if (str_replace(' ', '', $k) === strtoupper('FUNC:FIND_IN_SET')) {
                    if (isset($v[0]['column'])) $arg1 = $v[0]['column'];
                    if (isset($v[0]['value'])) {
                        $arg1 = self::mkNamedKey();
                        $this->setParams($arg1, is_array($v[0]['value']) ? implode(',', $v[0]['value']) : $v[0]['value']);
                    }

                    if (isset($v[1]['column'])) $arg2 = $v[1]['column'];
                    if (isset($v[1]['value'])) {
                        $arg2 = self::mkNamedKey();
                        $this->setParams($arg2, is_array($v[1]['value']) ? implode(',', $v[1]['value']) : $v[1]['value']);
                    }

                    $this->where .= " FIND_IN_SET($arg1, $arg2) ";
                    continue;
                }
            }

            if (is_array($v)) {
                // eg) 'column_name' => array('like', "%{$value}%")
                $named_key = self::mkNamedKey();
                $this->where .= " {$k} {$v[0]} {$named_key} ";
                $this->setParams($named_key, $v[1]);
            } else {
                // eg) 'column_name' => $value
                $named_key = self::mkNamedKey();
                $this->where .= " {$k} = {$named_key} ";
                $this->setParams($named_key, $v);
            }
        }
    }

    /**
     * すでに where 句 が定義されていた場合のみ、演算子を適用する
     *
     *  @param str $operator 演算子
     */
    public function ifDefineWhere($operator)
    {
        if ($this->where !== '')  $this->where .= ' '.strtoupper($operator).' ';
    }

    /**
     * where in 句
     *
     *  - 廃止予定 -
     */
    /*public function whereIn($column, $arr)
    {
        $buff = array();
        foreach ($arr as $k => $v) {
            $named_key = self::mkNamedKey();
            $buff[] = $named_key;
            $this->setParams($named_key, $v);
        }
        $this->wherein = "WHERE $column IN (".implode(', ', $buff).')';
    }*/

    /**
     * group by 句
     */
    public function groupBy()
    {
        $args = func_get_args();
        $this->groupby = 'GROUP BY '.implode(', ', $args);
    }

    /**
     * having 句
     */
    public function having()
    {
        // 未実装
    }

    /**
     * order by 句
     */
    public function orderBy($arr)
    {
        $this->orderby = 'ORDER BY ';

        if (isset($arr[0]) && strtoupper(trim($arr[0])) === 'RAND()') {
            $this->orderby .= 'rand()';
        } else {
            $buff = array();
            foreach ($arr as $k => $v) $buff[] = "$k $v";
            $this->orderby .= implode(', ', $buff);
        }
    }

    /**
     * limit 句
     */
    public function limit($offset, $limit)
    {
        $this->limit = "LIMIT $offset, $limit";
    }

    /**
     * values 句 (for insert)
     */
    public function values($arr)
    {
        $buff = array();
        foreach ($arr as $k => $v) {
            $buff[] = ":$k";
            $this->setParams($k, $v);
        }
        $this->values = implode(', ', $buff);
    }

    /**
     * set 句 (for update)
     */
    public function set($arr)
    {
        $buff = array();
        foreach ($arr as $k => $v) {
            if (isset($v[0]) && isset($v[1]) && $v[0] === '=') {
                // ex) 'count' => array('=', 'count + 1'), 
                $buff[] = "$k $v[0] $v[1]";
            } else {
                $buff[] = "$k = :{$k}";
                $this->setParams($k, $v);
            }
        }

        if ($this->set === '') {
            $this->set = implode(', ', $buff);
        } else {
            $this->set .= ', '.implode(', ', $buff);
        }
    }

    /**
     * ON DUPLICATE KEY UPDATE 句 (for insert)
     */
    public function onDuplicateKeyUpdate(array $params = array())
    {
        $buff = array();
        foreach ($params as $k => $v) {
            if (is_array($v)) {
                $buff[] = "{$k} {$v[0]} {$v[1]}";
            } else {
                $named_key = self::mkNamedKey();
                $buff[] = "{$k} = {$named_key}";
                $this->setParams($named_key, $v);
            }
        }
        $params = implode(', ', $buff);
        $this->duplicate_key_update = "ON DUPLICATE KEY UPDATE {$params}";
    }

    /**
     * FOR UPDATE 句
     */
    public function forUpdate()
    {
        $this->for_update = 'FOR UPDATE';
    }

    /**
     * LOCK TABLES 句
     */
    public function lock($mode)
    {
        return self::$pdo->exec("LOCK TABLES {$this->table} {$mode}");
    }

    /**
     * UNLOCK TABLES 句
     */
    public function unlock()
    {
        return self::$pdo->exec('UNLOCK TABLES');
    }

    /**
     * PDO::prepare メソッドを介さずに 生のクエリ を発行する。
     *
     * @param  str $type クエリタイプ (fetchColum|fetch|fechAll|execute)
     * @param  str $sql  クエリ
     * @return mixed
     * @access public
     */
    public function rawQuery($type, $sql)
    {
        switch ($type) {
            case 'fetchColmun':
                return self::$pdo->fetchColmun($sql, array());
                break;

            case 'fetch':
                return self::$pdo->fetch($sql, array());
                break;

            case 'fetchAll':
                return self::$pdo->fetchAll($sql, array());
                break;

            case 'execute':
                return self::$pdo->execute($sql, array());
                break;

            default:
                return '';
        }
    }

    /**
     * foundRows() if fetchAll.
     */
    public function foundRows()
    {
        return self::$pdo->foundRows();
    }

    /**
     * lastInsertId
     */
    public function lastInsertId($name = NULL)
    {
        return self::$pdo->lastInsertId($name);
    }

    /**
     * begin
     */
    public function begin()
    {
        return self::$pdo->begin();
    }

    /**
     * commit
     */
    public function commit()
    {
        return self::$pdo->commit();
    }

    /**
     * rollback
     */
    public function rollback()
    {
        return self::$pdo->rollback();
    }



    public function getQuery()
    {
        return $this->query;
    }
    public function getParams()
    {
        return $this->params;
    }

    public function fetchAll($execute = true)
    {
        $this->setQuery(self::SELECT);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->fetchAll($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function fetch($execute = true)
    {
        $this->setQuery(self::SELECT);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->fetch($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function fetchColumn($execute = true)
    {
        $this->setQuery(self::SELECT);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->fetchColumn($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function insert($execute = true)
    {
        $this->setQuery(self::INSERT);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->execute($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function update($execute = true)
    {
        $this->setQuery(self::UPDATE);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->execute($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function updateWithAffectedRows($execute = true)
    {
        $this->setQuery(self::UPDATE);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->executeWithAffectedRows($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function delete($execute = true)
    {
        $this->setQuery(self::DELETE);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->execute($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function deleteWithAffectedRows($execute = true)
    {
        $this->setQuery(self::DELETE);
        if ( ! $execute) return $this->dump();
        $result = self::$pdo->executeWithAffectedRows($this->getQuery(), $this->getParams());
        $this->initParams();
        return $result;
    }

    public function hasWhere()
    {
        return (bool)$this->where;
    }



    protected static function mkNamedKey()
    {
        static $i = 0;
        return ':' . $i++;
    }

    protected function setParams($named_key, $val)
    {
        if ( ! isset($this->params[$named_key])) $this->params[$named_key] = $val;
    }

    protected function getWhere()
    {
        //return ($this->wherein) ? $this->wherein : $this->where;
        return $this->where;
    }

    protected function setQuery($type)
    {
        switch ($type) {
            case self::SELECT:
                $this->query  = "SELECT $this->columns FROM $this->table $this->alias ";
                foreach (array_keys($this->join) as $k) {
                    $this->query .= $this->join[$k];
                    if (isset($this->using[$k])) $this->query .= $this->using[$k];
                }
                $this->query .= $this->getWhere()." $this->groupby $this->having $this->orderby $this->limit $this->for_update ";
                break;

            case self::INSERT:
                $this->query = "INSERT INTO $this->table ($this->columns) VALUES ($this->values) $this->duplicate_key_update ";
                break;

            case self::UPDATE:
                $this->query = "UPDATE $this->table SET $this->set ".$this->getWhere();
                break;

            case self::DELETE:
                $this->query = "DELETE FROM $this->table ".$this->getWhere();
                break;

            default:
                $this->query = '';
        }
    }

    /**
     * for debug.
     */
    protected function dump()
    {
        echo "<pre>\n";
        var_dump($this->getQuery(), $this->getParams());
        $this->_setLastQuery();
        var_dump($this->getLastQuery());
        exit;
    }

    /**
     * Set Last Query
     * @return void
     * @access private
     */
    private function _setLastQuery()
    {
        $pdo = self::$pdo->getPDO();
        $query = $this->getQuery();

        // 置換ミス防止のため、name_key の末尾に : を付与する
        // 例) :0 -> :0:
        $query = preg_replace('/:[0-9a-zA-Z_]+/', '\\0:', $query);

        $param = $this->getParams();
        foreach ($param as $key => $val) {
            $query = str_replace($key.':', $pdo->quote($val), $query);
        }
        $this->_last_query = $query;
    }

    /**
     * Get Last Query
     * @return string last query
     * @access public
     */
    public function getLastQuery()
    {
        return $this->_last_query;
    }

} //-- End of class


